//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3BiosealDateTime_h
#define _id3BiosealDateTime_h

#include <id3Bioseal/id3BiosealExports.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a date/time.
 */

/**
 * Handle to a DateTime object of the Bioseal module.
 */
typedef struct id3BiosealDateTime *ID3_BIOSEAL_DATE_TIME;

/**
 * @brief Initializes a DateTime object.
 * @param phDateTime [in] Handle to the DateTime object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDateTime_Initialize(ID3_BIOSEAL_DATE_TIME *phDateTime);

/**
 * @brief Releases the memory allocated for a DateTime.
 * @param phDateTime [in] Handle to the DateTime object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDateTime_Dispose(ID3_BIOSEAL_DATE_TIME *phDateTime);

/**
 * @brief Checks if the handle object is valid.
 * @param hDateTime [in] Handle to the DateTime object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDateTime_CheckHandle(ID3_BIOSEAL_DATE_TIME hDateTime);

/**
 * Gets the the year (1 through 9999, 0 if invalid date).
 * @param hDateTime [in] Handle to the DateTime object.
 * @param year [out] The year (1 through 9999, 0 if invalid date).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDateTime_GetYear(ID3_BIOSEAL_DATE_TIME hDateTime, int *year);

/**
 * Gets the the month (1 through 12).
 * @param hDateTime [in] Handle to the DateTime object.
 * @param month [out] The month (1 through 12).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDateTime_GetMonth(ID3_BIOSEAL_DATE_TIME hDateTime, int *month);

/**
 * Gets the the day (1 through the number of days in month).
 * @param hDateTime [in] Handle to the DateTime object.
 * @param day [out] The day (1 through the number of days in month).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDateTime_GetDay(ID3_BIOSEAL_DATE_TIME hDateTime, int *day);

/**
 * Gets the the hour (0 through 23, -1 if invalid time).
 * @param hDateTime [in] Handle to the DateTime object.
 * @param hour [out] The hour (0 through 23, -1 if invalid time).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDateTime_GetHour(ID3_BIOSEAL_DATE_TIME hDateTime, int *hour);

/**
 * Gets the the minute (0 through 59).
 * @param hDateTime [in] Handle to the DateTime object.
 * @param minute [out] The minute (0 through 59).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDateTime_GetMinute(ID3_BIOSEAL_DATE_TIME hDateTime, int *minute);

/**
 * Gets the the second (0 through 59).
 * @param hDateTime [in] Handle to the DateTime object.
 * @param second [out] The second (0 through 59).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDateTime_GetSecond(ID3_BIOSEAL_DATE_TIME hDateTime, int *second);

/**
 * Indicates whether the date and/or time is valid.
 * @param hDateTime [in] Handle to the DateTime object.
 * @param result [out] true if date and/or time is valid.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDateTime_IsValid(ID3_BIOSEAL_DATE_TIME hDateTime, bool *result);

/**
 * Indicates whether the time is valid.
 * @param hDateTime [in] Handle to the DateTime object.
 * @param result [out] true if the time is valid.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDateTime_IsTimeValid(ID3_BIOSEAL_DATE_TIME hDateTime, bool *result);

/**
 * Indicates whether the date is valid.
 * @param hDateTime [in] Handle to the DateTime object.
 * @param result [out] true if the date is valid.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDateTime_IsDateValid(ID3_BIOSEAL_DATE_TIME hDateTime, bool *result);

/**
 * Indicates whether the date and time are valid.
 * @param hDateTime [in] Handle to the DateTime object.
 * @param result [out] true if date and time is valid.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDateTime_IsDateTimeValid(ID3_BIOSEAL_DATE_TIME hDateTime, bool *result);

/**
 * Convert the date/time to a string.
 * @param hDateTime [in] Handle to the DateTime object.
 * @param result [out] Automatic format is YYYY-MM-DDTHH:MM:SS, YYYY-MM-DD or HH:MM:SS
 * @param resultSize [in,out] Size of the 'result' buffer. Receives the minimum required buffer size to hold the 'result' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'result' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDateTime_ToString(ID3_BIOSEAL_DATE_TIME hDateTime, char *result, int *resultSize);

#ifdef __cplusplus
}
#endif

#endif
