//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3BiosealDecryptionArgs_h
#define _id3BiosealDecryptionArgs_h

#include <id3Bioseal/id3BiosealExports.h>
#include <id3Bioseal/id3BiosealStringArray.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents the arguments to the decryption callback.
 */

/**
 * Handle to a DecryptionArgs object of the Bioseal module.
 */
typedef struct id3BiosealDecryptionArgs *ID3_BIOSEAL_DECRYPTION_ARGS;

/**
 * @brief Initializes a DecryptionArgs object.
 * @param phDecryptionArgs [in] Handle to the DecryptionArgs object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDecryptionArgs_Initialize(ID3_BIOSEAL_DECRYPTION_ARGS *phDecryptionArgs);

/**
 * @brief Releases the memory allocated for a DecryptionArgs.
 * @param phDecryptionArgs [in] Handle to the DecryptionArgs object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDecryptionArgs_Dispose(ID3_BIOSEAL_DECRYPTION_ARGS *phDecryptionArgs);

/**
 * @brief Checks if the handle object is valid.
 * @param hDecryptionArgs [in] Handle to the DecryptionArgs object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDecryptionArgs_CheckHandle(ID3_BIOSEAL_DECRYPTION_ARGS hDecryptionArgs);

/**
 * Gets the the name of the field to decrypt.
 * @param hDecryptionArgs [in] Handle to the DecryptionArgs object.
 * @param fieldName [out] The name of the field to decrypt.
 * @param fieldNameSize [in,out] Size of the 'fieldName' buffer. Receives the minimum required buffer size to hold the 'fieldName' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'fieldName' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDecryptionArgs_GetFieldName(ID3_BIOSEAL_DECRYPTION_ARGS hDecryptionArgs, char *fieldName, int *fieldNameSize);

/**
 * Gets the the data to decrypt as a binary array.
 * @param hDecryptionArgs [in] Handle to the DecryptionArgs object.
 * @param inputData [out] The data to decrypt as a binary array.
 * @param inputDataSize [in,out] Size of the 'inputData' buffer. Receives the number of elements in the 'inputData' array.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDecryptionArgs_GetInputData(ID3_BIOSEAL_DECRYPTION_ARGS hDecryptionArgs, unsigned char *inputData, int *inputDataSize);

/**
 * Gets the the decrypted data as a binary array.
 * @param hDecryptionArgs [in] Handle to the DecryptionArgs object.
 * @param outputData [out] The decrypted data as a binary array.
 * @param outputDataSize [in,out] Size of the 'outputData' buffer. Receives the number of elements in the 'outputData' array.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDecryptionArgs_GetOutputData(ID3_BIOSEAL_DECRYPTION_ARGS hDecryptionArgs, unsigned char *outputData, int *outputDataSize);

/**
 * Sets the the decrypted data as a binary array.
 * @param hDecryptionArgs [in] Handle to the DecryptionArgs object.
 * @param outputData [in] The decrypted data as a binary array.
 * @param outputDataSize [in] Size of the 'outputData' buffer.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDecryptionArgs_SetOutputData(ID3_BIOSEAL_DECRYPTION_ARGS hDecryptionArgs, unsigned char *outputData, int outputDataSize);

/**
 * Gets the gets a string list containing the keys in the dict.
 * @param hDecryptionArgs [in] Handle to the DecryptionArgs object.
 * @param hKeys [out] Gets a string list containing the keys in the dict.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDecryptionArgs_GetKeys(ID3_BIOSEAL_DECRYPTION_ARGS hDecryptionArgs, ID3_BIOSEAL_STRING_ARRAY hKeys);

/**
 * Gets an item of the DecryptionArgs object.
 * @param hDecryptionArgs [in] Handle to the DecryptionArgs object.
 * @param key [in] Unique key of the  item to get.
 * @param item [out]  item to get.
 * @param itemSize [in,out] Size of the 'item' buffer. Receives the minimum required buffer size to hold the 'item' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'item' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDecryptionArgs_Get(ID3_BIOSEAL_DECRYPTION_ARGS hDecryptionArgs, const char *key, char *item, int *itemSize);

/**
 * Gets the number of elements in the DecryptionArgs object.
 * @param hDecryptionArgs [in] Handle to the DecryptionArgs object.
 * @param count [out] Number of elements in the  object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDecryptionArgs_GetCount(ID3_BIOSEAL_DECRYPTION_ARGS hDecryptionArgs, int *count);

/**
 * Determines whether the DecryptionArgsobject contains the specified key.
 * @param hDecryptionArgs [in] Handle to the DecryptionArgs object.
 * @param key [in] The key to locate in the  object.
 * @param result [out] true if the  object contains an element with the specified key, otherwise false.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealDecryptionArgs_ContainsKey(ID3_BIOSEAL_DECRYPTION_ARGS hDecryptionArgs, const char *key, bool *result);

#ifdef __cplusplus
}
#endif

#endif
