//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3BiosealLogItem_h
#define _id3BiosealLogItem_h

#include <id3Bioseal/id3BiosealExports.h>
#include <id3Bioseal/id3BiosealLogItemType.h>
#include <id3Bioseal/id3BiosealError.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a log item.
 */

/**
 * Handle to a LogItem object of the Bioseal module.
 */
typedef struct id3BiosealLogItem *ID3_BIOSEAL_LOG_ITEM;

/**
 * @brief Initializes a LogItem object.
 * @param phLogItem [in] Handle to the LogItem object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLogItem_Initialize(ID3_BIOSEAL_LOG_ITEM *phLogItem);

/**
 * @brief Releases the memory allocated for a LogItem.
 * @param phLogItem [in] Handle to the LogItem object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLogItem_Dispose(ID3_BIOSEAL_LOG_ITEM *phLogItem);

/**
 * @brief Checks if the handle object is valid.
 * @param hLogItem [in] Handle to the LogItem object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLogItem_CheckHandle(ID3_BIOSEAL_LOG_ITEM hLogItem);

/**
 * Gets the the timestamp of the message.
 * @param hLogItem [in] Handle to the LogItem object.
 * @param timestamp [out] The timestamp of the message.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLogItem_GetTimestamp(ID3_BIOSEAL_LOG_ITEM hLogItem, long long *timestamp);

/**
 * Gets the the log message.
 * @param hLogItem [in] Handle to the LogItem object.
 * @param message [out] The log message.
 * @param messageSize [in,out] Size of the 'message' buffer. Receives the minimum required buffer size to hold the 'message' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'message' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLogItem_GetMessage(ID3_BIOSEAL_LOG_ITEM hLogItem, char *message, int *messageSize);

/**
 * Gets the the log type.
 * @param hLogItem [in] Handle to the LogItem object.
 * @param eItemType [out] The log type.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLogItem_GetItemType(ID3_BIOSEAL_LOG_ITEM hLogItem, id3BiosealLogItemType *eItemType);

/**
 * Gets the the log item error code if any.
 * @param hLogItem [in] Handle to the LogItem object.
 * @param eErrorCode [out] The log item error code if any.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLogItem_GetErrorCode(ID3_BIOSEAL_LOG_ITEM hLogItem, id3BiosealError *eErrorCode);

#ifdef __cplusplus
}
#endif

#endif
