//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3BiosealResourceCallbackArgs_h
#define _id3BiosealResourceCallbackArgs_h

#include <id3Bioseal/id3BiosealExports.h>
#include <id3Bioseal/id3BiosealResourceType.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Contains resource callback data.
 */

/**
 * Handle to a ResourceCallbackArgs object of the Bioseal module.
 */
typedef struct id3BiosealResourceCallbackArgs *ID3_BIOSEAL_RESOURCE_CALLBACK_ARGS;

/**
 * @brief Initializes a ResourceCallbackArgs object.
 * @param phResourceCallbackArgs [in] Handle to the ResourceCallbackArgs object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealResourceCallbackArgs_Initialize(ID3_BIOSEAL_RESOURCE_CALLBACK_ARGS *phResourceCallbackArgs);

/**
 * @brief Releases the memory allocated for a ResourceCallbackArgs.
 * @param phResourceCallbackArgs [in] Handle to the ResourceCallbackArgs object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealResourceCallbackArgs_Dispose(ID3_BIOSEAL_RESOURCE_CALLBACK_ARGS *phResourceCallbackArgs);

/**
 * @brief Checks if the handle object is valid.
 * @param hResourceCallbackArgs [in] Handle to the ResourceCallbackArgs object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealResourceCallbackArgs_CheckHandle(ID3_BIOSEAL_RESOURCE_CALLBACK_ARGS hResourceCallbackArgs);

/**
 * Gets the the URI used to retrieve the external resource.
 * @param hResourceCallbackArgs [in] Handle to the ResourceCallbackArgs object.
 * @param uri [out] The URI used to retrieve the external resource.
 * @param uriSize [in,out] Size of the 'uri' buffer. Receives the minimum required buffer size to hold the 'uri' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'uri' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealResourceCallbackArgs_GetUri(ID3_BIOSEAL_RESOURCE_CALLBACK_ARGS hResourceCallbackArgs, char *uri, int *uriSize);

/**
 * Gets the the resource name corresponing to the uri.
 * @param hResourceCallbackArgs [in] Handle to the ResourceCallbackArgs object.
 * @param resourceName [out] The resource name corresponing to the uri.
 * @param resourceNameSize [in,out] Size of the 'resourceName' buffer. Receives the minimum required buffer size to hold the 'resourceName' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'resourceName' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealResourceCallbackArgs_GetResourceName(ID3_BIOSEAL_RESOURCE_CALLBACK_ARGS hResourceCallbackArgs, char *resourceName, int *resourceNameSize);

/**
 * Gets the the resource type corresponding to the uri.
 * @param hResourceCallbackArgs [in] Handle to the ResourceCallbackArgs object.
 * @param eResourceType [out] The resource type corresponding to the uri.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealResourceCallbackArgs_GetResourceType(ID3_BIOSEAL_RESOURCE_CALLBACK_ARGS hResourceCallbackArgs, id3BiosealResourceType *eResourceType);

/**
 * Gets the the external resource data must be updated.
 * @param hResourceCallbackArgs [in] Handle to the ResourceCallbackArgs object.
 * @param requiresUpdate [out] The external resource data must be updated.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealResourceCallbackArgs_GetRequiresUpdate(ID3_BIOSEAL_RESOURCE_CALLBACK_ARGS hResourceCallbackArgs, bool *requiresUpdate);

/**
 * Gets the the external resource data as a byte array.
 * @param hResourceCallbackArgs [in] Handle to the ResourceCallbackArgs object.
 * @param outputData [out] The external resource data as a byte array.
 * @param outputDataSize [in,out] Size of the 'outputData' buffer. Receives the number of elements in the 'outputData' array.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealResourceCallbackArgs_GetOutputData(ID3_BIOSEAL_RESOURCE_CALLBACK_ARGS hResourceCallbackArgs, unsigned char *outputData, int *outputDataSize);

/**
 * Sets the the external resource data as a byte array.
 * @param hResourceCallbackArgs [in] Handle to the ResourceCallbackArgs object.
 * @param outputData [in] The external resource data as a byte array.
 * @param outputDataSize [in] Size of the 'outputData' buffer.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealResourceCallbackArgs_SetOutputData(ID3_BIOSEAL_RESOURCE_CALLBACK_ARGS hResourceCallbackArgs, unsigned char *outputData, int outputDataSize);

/**
 * Downloads a resource.
 * @param hResourceCallbackArgs [in] Handle to the ResourceCallbackArgs object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealResourceCallbackArgs_Download(ID3_BIOSEAL_RESOURCE_CALLBACK_ARGS hResourceCallbackArgs);

/**
 * Downloads a file from the specified URL and returns a string.
 * @param hResourceCallbackArgs [in] Handle to the ResourceCallbackArgs object.
 * @param url [in] The URL to the file.
 * @param data [out] The file as a string.
 * @param dataSize [in,out] Size of the 'data' buffer. Receives the minimum required buffer size to hold the 'data' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'data' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealResourceCallbackArgs_DownloadStringFile(ID3_BIOSEAL_RESOURCE_CALLBACK_ARGS hResourceCallbackArgs, const char *url, char *data, int *dataSize);

/**
 * Computes the SHA-256 of the external resource's data.
 * @param hResourceCallbackArgs [in] Handle to the ResourceCallbackArgs object.
 * @param result [out] A string that contains the SHA-256.
 * @param resultSize [in,out] Size of the 'result' buffer. Receives the minimum required buffer size to hold the 'result' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'result' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealResourceCallbackArgs_ComputeSha256(ID3_BIOSEAL_RESOURCE_CALLBACK_ARGS hResourceCallbackArgs, char *result, int *resultSize);

#ifdef __cplusplus
}
#endif

#endif
