//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3BiosealBarcodeReader_h
#define _id3BiosealBarcodeReader_h

#include <id3Bioseal/id3BiosealExports.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Provides functionality to decode barcodes from raw image data.
 */

/**
 * Handle to a BarcodeReader object of the Bioseal module.
 */
typedef struct id3BiosealBarcodeReader *ID3_BIOSEAL_BARCODE_READER;

/**
 * @brief Initializes a BarcodeReader object.
 * @param phBarcodeReader [in] Handle to the BarcodeReader object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealBarcodeReader_Initialize(ID3_BIOSEAL_BARCODE_READER *phBarcodeReader);

/**
 * @brief Releases the memory allocated for a BarcodeReader.
 * @param phBarcodeReader [in] Handle to the BarcodeReader object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealBarcodeReader_Dispose(ID3_BIOSEAL_BARCODE_READER *phBarcodeReader);

/**
 * @brief Checks if the handle object is valid.
 * @param hBarcodeReader [in] Handle to the BarcodeReader object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealBarcodeReader_CheckHandle(ID3_BIOSEAL_BARCODE_READER hBarcodeReader);

/**
 * Gets the holds the decoded data extracted from the barcode.
 * @param hBarcodeReader [in] Handle to the BarcodeReader object.
 * @param outputData [out] Holds the decoded data extracted from the barcode.
 * @param outputDataSize [in,out] Size of the 'outputData' buffer. Receives the number of elements in the 'outputData' array.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealBarcodeReader_GetOutputData(ID3_BIOSEAL_BARCODE_READER hBarcodeReader, unsigned char *outputData, int *outputDataSize);

/**
 * Attempts to decode a barcode from the provided raw image data.
 * @param hBarcodeReader [in] Handle to the BarcodeReader object.
 * @param data [in] An array of bytes representing the raw image data that contains the barcode to be decoded.
 * @param dataSize [in] Size of the 'data' buffer.
 * @param result [out] A boolean output that indicates the success of the decoding process. It is set to 'true' if the barcode is successfully decoded, otherwise 'false'.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealBarcodeReader_Decode(ID3_BIOSEAL_BARCODE_READER hBarcodeReader, unsigned char *data, int dataSize, bool *result);

#ifdef __cplusplus
}
#endif

#endif
