//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3BiosealCertificateInformation_h
#define _id3BiosealCertificateInformation_h

#include <id3Bioseal/id3BiosealExports.h>
#include <id3Bioseal/id3BiosealDateTime.h>
#include <id3Bioseal/id3BiosealStringArray.h>
#include <id3Bioseal/id3BiosealCertificateExtensionList.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents the information about a certificate.
 */

/**
 * Handle to a CertificateInformation object of the Bioseal module.
 */
typedef struct id3BiosealCertificateInformation *ID3_BIOSEAL_CERTIFICATE_INFORMATION;

/**
 * @brief Initializes a CertificateInformation object.
 * @param phCertificateInformation [in] Handle to the CertificateInformation object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_Initialize(ID3_BIOSEAL_CERTIFICATE_INFORMATION *phCertificateInformation);

/**
 * @brief Releases the memory allocated for a CertificateInformation.
 * @param phCertificateInformation [in] Handle to the CertificateInformation object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_Dispose(ID3_BIOSEAL_CERTIFICATE_INFORMATION *phCertificateInformation);

/**
 * @brief Checks if the handle object is valid.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_CheckHandle(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation);

/**
 * Gets the the common name (CN) of the certificate issuer.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param issuerCommonName [out] The common name (CN) of the certificate issuer.
 * @param issuerCommonNameSize [in,out] Size of the 'issuerCommonName' buffer. Receives the minimum required buffer size to hold the 'issuerCommonName' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'issuerCommonName' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetIssuerCommonName(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, char *issuerCommonName, int *issuerCommonNameSize);

/**
 * Gets the the organization (O) of the certificate issuer.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param issuerOrganization [out] The organization (O) of the certificate issuer.
 * @param issuerOrganizationSize [in,out] Size of the 'issuerOrganization' buffer. Receives the minimum required buffer size to hold the 'issuerOrganization' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'issuerOrganization' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetIssuerOrganization(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, char *issuerOrganization, int *issuerOrganizationSize);

/**
 * Gets the the country (C) of the certificate issuer.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param issuerCountry [out] The country (C) of the certificate issuer.
 * @param issuerCountrySize [in,out] Size of the 'issuerCountry' buffer. Receives the minimum required buffer size to hold the 'issuerCountry' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'issuerCountry' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetIssuerCountry(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, char *issuerCountry, int *issuerCountrySize);

/**
 * Gets the the organizational unit (OU) of the certificate issuer.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param issuerOrganizationalUnit [out] The organizational unit (OU) of the certificate issuer.
 * @param issuerOrganizationalUnitSize [in,out] Size of the 'issuerOrganizationalUnit' buffer. Receives the minimum required buffer size to hold the 'issuerOrganizationalUnit' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'issuerOrganizationalUnit' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetIssuerOrganizationalUnit(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, char *issuerOrganizationalUnit, int *issuerOrganizationalUnitSize);

/**
 * Gets the the distinguished name (DN) of the certificate issuer.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param issuerDistinguishedName [out] The distinguished name (DN) of the certificate issuer.
 * @param issuerDistinguishedNameSize [in,out] Size of the 'issuerDistinguishedName' buffer. Receives the minimum required buffer size to hold the 'issuerDistinguishedName' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'issuerDistinguishedName' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetIssuerDistinguishedName(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, char *issuerDistinguishedName, int *issuerDistinguishedNameSize);

/**
 * Gets the the subject common name (CN) from the certificate.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param subjectCommonName [out] The subject common name (CN) from the certificate.
 * @param subjectCommonNameSize [in,out] Size of the 'subjectCommonName' buffer. Receives the minimum required buffer size to hold the 'subjectCommonName' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'subjectCommonName' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetSubjectCommonName(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, char *subjectCommonName, int *subjectCommonNameSize);

/**
 * Gets the the organization (O) of the certificate subject.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param subjectOrganization [out] The organization (O) of the certificate subject.
 * @param subjectOrganizationSize [in,out] Size of the 'subjectOrganization' buffer. Receives the minimum required buffer size to hold the 'subjectOrganization' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'subjectOrganization' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetSubjectOrganization(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, char *subjectOrganization, int *subjectOrganizationSize);

/**
 * Gets the the country (C) of the certificate subject.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param subjectCountry [out] The country (C) of the certificate subject.
 * @param subjectCountrySize [in,out] Size of the 'subjectCountry' buffer. Receives the minimum required buffer size to hold the 'subjectCountry' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'subjectCountry' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetSubjectCountry(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, char *subjectCountry, int *subjectCountrySize);

/**
 * Gets the the organizational unit (OU) of the certificate subject.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param subjectOrganizationalUnit [out] The organizational unit (OU) of the certificate subject.
 * @param subjectOrganizationalUnitSize [in,out] Size of the 'subjectOrganizationalUnit' buffer. Receives the minimum required buffer size to hold the 'subjectOrganizationalUnit' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'subjectOrganizationalUnit' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetSubjectOrganizationalUnit(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, char *subjectOrganizationalUnit, int *subjectOrganizationalUnitSize);

/**
 * Gets the the subject distinguished (DN) name of the certificate.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param subjectDistinguishedName [out] The subject distinguished (DN) name of the certificate.
 * @param subjectDistinguishedNameSize [in,out] Size of the 'subjectDistinguishedName' buffer. Receives the minimum required buffer size to hold the 'subjectDistinguishedName' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'subjectDistinguishedName' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetSubjectDistinguishedName(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, char *subjectDistinguishedName, int *subjectDistinguishedNameSize);

/**
 * Gets the the date in local time when a certificate becomes valid.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param hNotBefore [out] The date in local time when a certificate becomes valid.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetNotBefore(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, ID3_BIOSEAL_DATE_TIME hNotBefore);

/**
 * Gets the the date in local time after which a certificate is no longer valid.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param hNotAfter [out] The date in local time after which a certificate is no longer valid.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetNotAfter(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, ID3_BIOSEAL_DATE_TIME hNotAfter);

/**
 * Gets the the serial number of the certificate.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param serialNumber [out] The serial number of the certificate.
 * @param serialNumberSize [in,out] Size of the 'serialNumber' buffer. Receives the number of elements in the 'serialNumber' array.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetSerialNumber(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, unsigned char *serialNumber, int *serialNumberSize);

/**
 * Gets the the serial number from the certificate as a string.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param serialNumberString [out] The serial number from the certificate as a string.
 * @param serialNumberStringSize [in,out] Size of the 'serialNumberString' buffer. Receives the minimum required buffer size to hold the 'serialNumberString' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'serialNumberString' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetSerialNumberString(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, char *serialNumberString, int *serialNumberStringSize);

/**
 * Gets the the certificate data.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param data [out] The certificate data.
 * @param dataSize [in,out] Size of the 'data' buffer. Receives the number of elements in the 'data' array.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetData(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, unsigned char *data, int *dataSize);

/**
 * Gets the the OID of the certificate's signing algorithm.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param signatureAlgorithm [out] The OID of the certificate's signing algorithm.
 * @param signatureAlgorithmSize [in,out] Size of the 'signatureAlgorithm' buffer. Receives the minimum required buffer size to hold the 'signatureAlgorithm' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'signatureAlgorithm' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetSignatureAlgorithm(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, char *signatureAlgorithm, int *signatureAlgorithmSize);

/**
 * Gets the the certificate version.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param version [out] The certificate version.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetVersion(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, int *version);

/**
 * Gets the the certificate public key.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param publicKey [out] The certificate public key.
 * @param publicKeySize [in,out] Size of the 'publicKey' buffer. Receives the number of elements in the 'publicKey' array.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetPublicKey(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, unsigned char *publicKey, int *publicKeySize);

/**
 * Gets the the certificate key usage list.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param hKeyUsage [out] The certificate key usage list.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetKeyUsage(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, ID3_BIOSEAL_STRING_ARRAY hKeyUsage);

/**
 * Gets the the certificate policies.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param hPolicies [out] The certificate policies.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetPolicies(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, ID3_BIOSEAL_STRING_ARRAY hPolicies);

/**
 * Gets the the certificate extension list.
 * @param hCertificateInformation [in] Handle to the CertificateInformation object.
 * @param hCertificateExtensionList [out] The certificate extension list.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealCertificateInformation_GetCertificateExtensionList(ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation, ID3_BIOSEAL_CERTIFICATE_EXTENSION_LIST hCertificateExtensionList);

#ifdef __cplusplus
}
#endif

#endif
