//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3BiosealExtension_h
#define _id3BiosealExtension_h

#include <id3Bioseal/id3BiosealExports.h>
#include <id3Bioseal/id3BiosealStringArray.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents an extension.
 */

/**
 * Handle to a Extension object of the Bioseal module.
 */
typedef struct id3BiosealExtension *ID3_BIOSEAL_EXTENSION;

/**
 * @brief Initializes a Extension object.
 * @param phExtension [in] Handle to the Extension object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtension_Initialize(ID3_BIOSEAL_EXTENSION *phExtension);

/**
 * @brief Releases the memory allocated for a Extension.
 * @param phExtension [in] Handle to the Extension object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtension_Dispose(ID3_BIOSEAL_EXTENSION *phExtension);

/**
 * @brief Checks if the handle object is valid.
 * @param hExtension [in] Handle to the Extension object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtension_CheckHandle(ID3_BIOSEAL_EXTENSION hExtension);

/**
 * Gets the the extension name.
 * @param hExtension [in] Handle to the Extension object.
 * @param name [out] The extension name.
 * @param nameSize [in,out] Size of the 'name' buffer. Receives the minimum required buffer size to hold the 'name' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'name' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtension_GetName(ID3_BIOSEAL_EXTENSION hExtension, char *name, int *nameSize);

/**
 * Gets the gets a string list containing the keys in the dict.
 * @param hExtension [in] Handle to the Extension object.
 * @param hKeys [out] Gets a string list containing the keys in the dict.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtension_GetKeys(ID3_BIOSEAL_EXTENSION hExtension, ID3_BIOSEAL_STRING_ARRAY hKeys);

/**
 * Gets an item of the Extension object.
 * @param hExtension [in] Handle to the Extension object.
 * @param key [in] Unique key of the  item to get.
 * @param item [out]  item to get.
 * @param itemSize [in,out] Size of the 'item' buffer. Receives the minimum required buffer size to hold the 'item' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'item' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtension_Get(ID3_BIOSEAL_EXTENSION hExtension, const char *key, char *item, int *itemSize);

/**
 * Gets the number of elements in the Extension object.
 * @param hExtension [in] Handle to the Extension object.
 * @param count [out] Number of elements in the  object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtension_GetCount(ID3_BIOSEAL_EXTENSION hExtension, int *count);

/**
 * Determines whether the Extensionobject contains the specified key.
 * @param hExtension [in] Handle to the Extension object.
 * @param key [in] The key to locate in the  object.
 * @param result [out] true if the  object contains an element with the specified key, otherwise false.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtension_ContainsKey(ID3_BIOSEAL_EXTENSION hExtension, const char *key, bool *result);

#ifdef __cplusplus
}
#endif

#endif
