//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3BiosealField_h
#define _id3BiosealField_h

#include <id3Bioseal/id3BiosealExports.h>
#include <id3Bioseal/id3BiosealFieldType.h>
#include <id3Bioseal/id3BiosealDateTime.h>
#include <id3Bioseal/id3BiosealStringArray.h>
#include <id3Bioseal/id3BiosealDateTimeArray.h>
#include <id3Bioseal/id3BiosealFieldExtensionType.h>
#include <id3Bioseal/id3BiosealExtension.h>
#include <id3Bioseal/id3BiosealVisibilityCondition.h>
#include <id3Bioseal/id3BiosealField.h>
#include <id3Bioseal/id3BiosealBiometricDataType.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a field or a dictionary of fields.
 */

/**
 * Handle to a Field object of the Bioseal module.
 */
typedef struct id3BiosealField *ID3_BIOSEAL_FIELD;

/**
 * @brief Initializes a Field object.
 * @param phField [in] Handle to the Field object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_Initialize(ID3_BIOSEAL_FIELD *phField);

/**
 * @brief Releases the memory allocated for a Field.
 * @param phField [in] Handle to the Field object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_Dispose(ID3_BIOSEAL_FIELD *phField);

/**
 * @brief Checks if the handle object is valid.
 * @param hField [in] Handle to the Field object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_CheckHandle(ID3_BIOSEAL_FIELD hField);

/**
 * Gets the the field type.
 * @param hField [in] Handle to the Field object.
 * @param eFieldType [out] The field type.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetFieldType(ID3_BIOSEAL_FIELD hField, id3BiosealFieldType *eFieldType);

/**
 * Gets the the field name.
 * @param hField [in] Handle to the Field object.
 * @param name [out] The field name.
 * @param nameSize [in,out] Size of the 'name' buffer. Receives the minimum required buffer size to hold the 'name' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'name' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetName(ID3_BIOSEAL_FIELD hField, char *name, int *nameSize);

/**
 * Gets the the field value as a 64-bit integer.
 * @param hField [in] Handle to the Field object.
 * @param valueAsInteger [out] The field value as a 64-bit integer.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsInteger(ID3_BIOSEAL_FIELD hField, long long *valueAsInteger);

/**
 * Gets the the field value as a boolean.
 * @param hField [in] Handle to the Field object.
 * @param valueAsBoolean [out] The field value as a boolean.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsBoolean(ID3_BIOSEAL_FIELD hField, bool *valueAsBoolean);

/**
 * Gets the the field value as a float.
 * @param hField [in] Handle to the Field object.
 * @param valueAsFloat [out] The field value as a float.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsFloat(ID3_BIOSEAL_FIELD hField, float *valueAsFloat);

/**
 * Gets the the field value as a string.
 * @param hField [in] Handle to the Field object.
 * @param valueAsString [out] The field value as a string.
 * @param valueAsStringSize [in,out] Size of the 'valueAsString' buffer. Receives the minimum required buffer size to hold the 'valueAsString' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'valueAsString' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsString(ID3_BIOSEAL_FIELD hField, char *valueAsString, int *valueAsStringSize);

/**
 * Gets the the data as a binary array.
 * @param hField [in] Handle to the Field object.
 * @param valueAsBinary [out] The data as a binary array.
 * @param valueAsBinarySize [in,out] Size of the 'valueAsBinary' buffer. Receives the number of elements in the 'valueAsBinary' array.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsBinary(ID3_BIOSEAL_FIELD hField, unsigned char *valueAsBinary, int *valueAsBinarySize);

/**
 * Gets the the field value as a date.
 * @param hField [in] Handle to the Field object.
 * @param hValueAsDate [out] The field value as a date.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsDate(ID3_BIOSEAL_FIELD hField, ID3_BIOSEAL_DATE_TIME hValueAsDate);

/**
 * Gets the the field value as a time.
 * @param hField [in] Handle to the Field object.
 * @param hValueAsTime [out] The field value as a time.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsTime(ID3_BIOSEAL_FIELD hField, ID3_BIOSEAL_DATE_TIME hValueAsTime);

/**
 * Gets the the field value as a date/time.
 * @param hField [in] Handle to the Field object.
 * @param hValueAsDateTime [out] The field value as a date/time.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsDateTime(ID3_BIOSEAL_FIELD hField, ID3_BIOSEAL_DATE_TIME hValueAsDateTime);

/**
 * Gets the the field value as a timestamp (without timezone computation).
 * @param hField [in] Handle to the Field object.
 * @param valueAsTimestamp [out] The field value as a timestamp (without timezone computation).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsTimestamp(ID3_BIOSEAL_FIELD hField, long long *valueAsTimestamp);

/**
 * Gets the the field value as an array of 64-bit integers.
 * @param hField [in] Handle to the Field object.
 * @param valueAsIntegerArray [out] The field value as an array of 64-bit integers.
 * @param valueAsIntegerArraySize [in,out] Size of the 'valueAsIntegerArray' buffer. Receives the number of elements in the 'valueAsIntegerArray' array.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsIntegerArray(ID3_BIOSEAL_FIELD hField, long long *valueAsIntegerArray, int *valueAsIntegerArraySize);

/**
 * Gets the the field value as an array of booleans.
 * @param hField [in] Handle to the Field object.
 * @param valueAsBooleanArray [out] The field value as an array of booleans.
 * @param valueAsBooleanArraySize [in,out] Size of the 'valueAsBooleanArray' buffer. Receives the number of elements in the 'valueAsBooleanArray' array.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsBooleanArray(ID3_BIOSEAL_FIELD hField, bool *valueAsBooleanArray, int *valueAsBooleanArraySize);

/**
 * Gets the the field value as an array of floats.
 * @param hField [in] Handle to the Field object.
 * @param valueAsFloatArray [out] The field value as an array of floats.
 * @param valueAsFloatArraySize [in,out] Size of the 'valueAsFloatArray' buffer. Receives the number of elements in the 'valueAsFloatArray' array.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsFloatArray(ID3_BIOSEAL_FIELD hField, float *valueAsFloatArray, int *valueAsFloatArraySize);

/**
 * Gets the the field value as an array of strings.
 * @param hField [in] Handle to the Field object.
 * @param hValueAsStringArray [out] The field value as an array of strings.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsStringArray(ID3_BIOSEAL_FIELD hField, ID3_BIOSEAL_STRING_ARRAY hValueAsStringArray);

/**
 * Gets the the field value as an array of dates.
 * @param hField [in] Handle to the Field object.
 * @param hValueAsDateTimeArray [out] The field value as an array of dates.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsDateTimeArray(ID3_BIOSEAL_FIELD hField, ID3_BIOSEAL_DATE_TIME_ARRAY hValueAsDateTimeArray);

/**
 * Gets the the field value as an array of timestamp.
 * @param hField [in] Handle to the Field object.
 * @param valueAsTimestampArray [out] The field value as an array of timestamp.
 * @param valueAsTimestampArraySize [in,out] Size of the 'valueAsTimestampArray' buffer. Receives the number of elements in the 'valueAsTimestampArray' array.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetValueAsTimestampArray(ID3_BIOSEAL_FIELD hField, long long *valueAsTimestampArray, int *valueAsTimestampArraySize);

/**
 * Gets the A value indicating whether if the field is null.
 * @param hField [in] Handle to the Field object.
 * @param isNull [out] A value indicating whether if the field is null.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetIsNull(ID3_BIOSEAL_FIELD hField, bool *isNull);

/**
 * Gets the A value indicating whether the field (or one of its child fields) contains biometric data.
 * @param hField [in] Handle to the Field object.
 * @param containsBiometrics [out] A value indicating whether the field (or one of its child fields) contains biometric data.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetContainsBiometrics(ID3_BIOSEAL_FIELD hField, bool *containsBiometrics);

/**
 * Gets the the field extension type.
 * @param hField [in] Handle to the Field object.
 * @param eExtensionType [out] The field extension type.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetExtensionType(ID3_BIOSEAL_FIELD hField, id3BiosealFieldExtensionType *eExtensionType);

/**
 * Gets the the field extension.
 * @param hField [in] Handle to the Field object.
 * @param hFieldExtension [out] The field extension.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetFieldExtension(ID3_BIOSEAL_FIELD hField, ID3_BIOSEAL_EXTENSION hFieldExtension);

/**
 * Gets the the visibility condition.
 * @param hField [in] Handle to the Field object.
 * @param eVisibilityCondition [out] The visibility condition.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetVisibilityCondition(ID3_BIOSEAL_FIELD hField, id3BiosealVisibilityCondition *eVisibilityCondition);

/**
 * Gets the gets a string list containing the keys in the dict.
 * @param hField [in] Handle to the Field object.
 * @param hKeys [out] Gets a string list containing the keys in the dict.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetKeys(ID3_BIOSEAL_FIELD hField, ID3_BIOSEAL_STRING_ARRAY hKeys);

/**
 * Gets an item of the Field object.
 * @param hField [in] Handle to the Field object.
 * @param key [in] Unique key of the Field item to get.
 * @param hFieldItem [out] Field item to get.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_Get(ID3_BIOSEAL_FIELD hField, const char *key, ID3_BIOSEAL_FIELD hFieldItem);

/**
 * Gets the number of elements in the Field object.
 * @param hField [in] Handle to the Field object.
 * @param count [out] Number of elements in the Field object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetCount(ID3_BIOSEAL_FIELD hField, int *count);

/**
 * Determines whether the Fieldobject contains the specified key.
 * @param hField [in] Handle to the Field object.
 * @param key [in] The key to locate in the Field object.
 * @param result [out] true if the Field object contains an element with the specified key, otherwise false.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_ContainsKey(ID3_BIOSEAL_FIELD hField, const char *key, bool *result);

/**
 * Gets the object at the specified index.
 * @param hField [in] Handle to the Field object.
 * @param index [in] The index of the object to retrieve.
 * @param hItem [out] The field that contains the object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetObjectAtIndex(ID3_BIOSEAL_FIELD hField, int index, ID3_BIOSEAL_FIELD hItem);

/**
 * Gets the binary data at the specified index.
 * @param hField [in] Handle to the Field object.
 * @param index [in] The index of the binary field to retrieve.
 * @param item [out] The binary data.
 * @param itemSize [in,out] Size of the 'item' buffer. Receives the number of elements in the 'item' array.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetBinaryAtIndex(ID3_BIOSEAL_FIELD hField, int index, unsigned char *item, int *itemSize);

/**
 * Set the time zone offset in seconds.
 * @param hField [in] Handle to the Field object.
 * @param offsetInSeconds [in] The time zone offset in seconds.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_SetTimeZoneOffset(ID3_BIOSEAL_FIELD hField, int offsetInSeconds);

/**
 * Returns a value indicating whether the field (or one of its child fields) contains an extension of the specified type.
 * @param hField [in] Handle to the Field object.
 * @param eExtensionType [in] The field extension type.
 * @param result [out] A value indicating whether the field (or one of its child fields) contains the specified extension.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_ContainsExtension(ID3_BIOSEAL_FIELD hField, id3BiosealFieldExtensionType eExtensionType, bool *result);

/**
 * Returns a value indicating whether the field (or one of its child fields) contains an extension with specified name.
 * @param hField [in] Handle to the Field object.
 * @param extensionName [in] A string that identifies the extension.
 * @param result [out] A value indicating whether the field (or one of its child fields) contains the specified extension.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_ContainsExtensionByName(ID3_BIOSEAL_FIELD hField, const char *extensionName, bool *result);

/**
 * Returns a value indicating whether the field (or one of its child fields) contains biometric data of the specified type and/or format.
 * @param hField [in] Handle to the Field object.
 * @param eBiometricDataType [in] The biometric data type.
 * @param biometricFormat [in] The biometric data format.
 * @param result [out] A value indicating whether the field (or one of its child fields) contains biometric data of the specified type and/or format.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_ContainsBiometricData(ID3_BIOSEAL_FIELD hField, id3BiosealBiometricDataType eBiometricDataType, const char *biometricFormat, bool *result);

/**
 * Retrieves the localized label based on the provided language code.
 * @param hField [in] Handle to the Field object.
 * @param language [in] The language code for which the label should be localized (e.g., 'en' for English, 'fr' for French).
 * @param label [out] The localized label corresponding to the provided language code.
 * @param labelSize [in,out] Size of the 'label' buffer. Receives the minimum required buffer size to hold the 'label' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'label' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetLocalizedLabel(ID3_BIOSEAL_FIELD hField, const char *language, char *label, int *labelSize);

/**
 * Retrieves the localized value based on the provided language code.
 * @param hField [in] Handle to the Field object.
 * @param language [in] The language code for which the value should be localized (e.g., 'en' for English, 'fr' for French).
 * @param value [out] The localized value corresponding to the provided language code.
 * @param valueSize [in,out] Size of the 'value' buffer. Receives the minimum required buffer size to hold the 'value' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'value' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealField_GetLocalizedValue(ID3_BIOSEAL_FIELD hField, const char *language, char *value, int *valueSize);

#ifdef __cplusplus
}
#endif

#endif
