//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3BiosealManifest_h
#define _id3BiosealManifest_h

#include <id3Bioseal/id3BiosealExports.h>
#include <id3Bioseal/id3BiosealStringArray.h>
#include <id3Bioseal/id3BiosealExtensionList.h>
#include <id3Bioseal/id3BiosealLog.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Provides information about the manifest.
 */

/**
 * Handle to a Manifest object of the Bioseal module.
 */
typedef struct id3BiosealManifest *ID3_BIOSEAL_MANIFEST;

/**
 * @brief Initializes a Manifest object.
 * @param phManifest [in] Handle to the Manifest object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealManifest_Initialize(ID3_BIOSEAL_MANIFEST *phManifest);

/**
 * @brief Releases the memory allocated for a Manifest.
 * @param phManifest [in] Handle to the Manifest object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealManifest_Dispose(ID3_BIOSEAL_MANIFEST *phManifest);

/**
 * @brief Checks if the handle object is valid.
 * @param hManifest [in] Handle to the Manifest object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealManifest_CheckHandle(ID3_BIOSEAL_MANIFEST hManifest);

/**
 * Gets the the manifest ID in ISO 22385 format.
 * @param hManifest [in] Handle to the Manifest object.
 * @param manifestId [out] The manifest ID in ISO 22385 format.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealManifest_GetManifestId(ID3_BIOSEAL_MANIFEST hManifest, int *manifestId);

/**
 * Gets the the manifest version.
 * @param hManifest [in] Handle to the Manifest object.
 * @param manifestVersion [out] The manifest version.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealManifest_GetManifestVersion(ID3_BIOSEAL_MANIFEST hManifest, int *manifestVersion);

/**
 * Gets the the list of supported languages declared in the manifest.
 * @param hManifest [in] Handle to the Manifest object.
 * @param hSupportedLanguages [out] The list of supported languages declared in the manifest.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealManifest_GetSupportedLanguages(ID3_BIOSEAL_MANIFEST hManifest, ID3_BIOSEAL_STRING_ARRAY hSupportedLanguages);

/**
 * Gets the the extension list.
 * @param hManifest [in] Handle to the Manifest object.
 * @param hExtensionList [out] The extension list.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealManifest_GetExtensionList(ID3_BIOSEAL_MANIFEST hManifest, ID3_BIOSEAL_EXTENSION_LIST hExtensionList);

/**
 * Gets the the log.
 * @param hManifest [in] Handle to the Manifest object.
 * @param hLog [out] The log.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealManifest_GetLog(ID3_BIOSEAL_MANIFEST hManifest, ID3_BIOSEAL_LOG hLog);

/**
 * Sets the the log.
 * @param hManifest [in] Handle to the Manifest object.
 * @param hLog [in] The log.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealManifest_SetLog(ID3_BIOSEAL_MANIFEST hManifest, ID3_BIOSEAL_LOG hLog);

/**
 * Loads the manifest.
 * @param hManifest [in] Handle to the Manifest object.
 * @param manifestData [in] The manifest data.
 * @param manifestDataSize [in] Size of the 'manifestData' buffer.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealManifest_Load(ID3_BIOSEAL_MANIFEST hManifest, unsigned char *manifestData, int manifestDataSize);

/**
 * Loads the manifest.
 * @param hManifest [in] Handle to the Manifest object.
 * @param manifestData [in] The manifest data.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealManifest_LoadFromString(ID3_BIOSEAL_MANIFEST hManifest, const char *manifestData);

/**
 * Loads the manifest from file.
 * @param hManifest [in] Handle to the Manifest object.
 * @param path [in] The manifest path.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealManifest_LoadFromFile(ID3_BIOSEAL_MANIFEST hManifest, const char *path);

/**
 * Gets the name of the document.
 * @param hManifest [in] Handle to the Manifest object.
 * @param language [in] The expected language.
 * @param documentName [out] The name of the document.
 * @param documentNameSize [in,out] Size of the 'documentName' buffer. Receives the minimum required buffer size to hold the 'documentName' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'documentName' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealManifest_GetDocumentName(ID3_BIOSEAL_MANIFEST hManifest, const char *language, char *documentName, int *documentNameSize);

/**
 * Gets the description of the document.
 * @param hManifest [in] Handle to the Manifest object.
 * @param language [in] The expected language.
 * @param documentDescription [out] The description of the document.
 * @param documentDescriptionSize [in,out] Size of the 'documentDescription' buffer. Receives the minimum required buffer size to hold the 'documentDescription' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'documentDescription' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealManifest_GetDocumentDescription(ID3_BIOSEAL_MANIFEST hManifest, const char *language, char *documentDescription, int *documentDescriptionSize);

#ifdef __cplusplus
}
#endif

#endif
