//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3BiosealUserAuthentication_h
#define _id3BiosealUserAuthentication_h

#include <id3Bioseal/id3BiosealExports.h>
#include <id3Bioseal/id3BiosealUserAuthenticationMethod.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents the user authentication parameters.
 */

/**
 * Handle to a UserAuthentication object of the Bioseal module.
 */
typedef struct id3BiosealUserAuthentication *ID3_BIOSEAL_USER_AUTHENTICATION;

/**
 * @brief Initializes a UserAuthentication object.
 * @param phUserAuthentication [in] Handle to the UserAuthentication object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealUserAuthentication_Initialize(ID3_BIOSEAL_USER_AUTHENTICATION *phUserAuthentication);

/**
 * @brief Releases the memory allocated for a UserAuthentication.
 * @param phUserAuthentication [in] Handle to the UserAuthentication object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealUserAuthentication_Dispose(ID3_BIOSEAL_USER_AUTHENTICATION *phUserAuthentication);

/**
 * @brief Checks if the handle object is valid.
 * @param hUserAuthentication [in] Handle to the UserAuthentication object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealUserAuthentication_CheckHandle(ID3_BIOSEAL_USER_AUTHENTICATION hUserAuthentication);

/**
 * Gets the the user authentication method.
 * @param hUserAuthentication [in] Handle to the UserAuthentication object.
 * @param eMethod [out] The user authentication method.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealUserAuthentication_GetMethod(ID3_BIOSEAL_USER_AUTHENTICATION hUserAuthentication, id3BiosealUserAuthenticationMethod *eMethod);

/**
 * Gets the the name of the payload field containing the password value.
 * @param hUserAuthentication [in] Handle to the UserAuthentication object.
 * @param passwordFieldName [out] The name of the payload field containing the password value.
 * @param passwordFieldNameSize [in,out] Size of the 'passwordFieldName' buffer. Receives the minimum required buffer size to hold the 'passwordFieldName' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'passwordFieldName' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealUserAuthentication_GetPasswordFieldName(ID3_BIOSEAL_USER_AUTHENTICATION hUserAuthentication, char *passwordFieldName, int *passwordFieldNameSize);

/**
 * Gets the the hash algorithm used to protect the password value. Possible values are: Sha256, Sha384, Sha512.
 * @param hUserAuthentication [in] Handle to the UserAuthentication object.
 * @param passwordHashAlgorithm [out] The hash algorithm used to protect the password value. Possible values are: Sha256, Sha384, Sha512.
 * @param passwordHashAlgorithmSize [in,out] Size of the 'passwordHashAlgorithm' buffer. Receives the minimum required buffer size to hold the 'passwordHashAlgorithm' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'passwordHashAlgorithm' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealUserAuthentication_GetPasswordHashAlgorithm(ID3_BIOSEAL_USER_AUTHENTICATION hUserAuthentication, char *passwordHashAlgorithm, int *passwordHashAlgorithmSize);

#ifdef __cplusplus
}
#endif

#endif
