//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------

/// Enumerates the error codes.
enum BiosealError {
	/// No error.
	success,
	/// The given buffer is too small.
	insufficientBuffer,
	/// The requested object does not exist; a 'null' should be returned if possible.
	objectDoesNotExist,
	/// Base error code for BioSeal module.
	base,
	/// Minimum error code for BioSeal module.
	minimum,
	/// The handle is invalid.
	invalidHandle,
	/// The parameter is invalid.
	invalidParameter,
	/// An exception was fired in the callback.
	exceptionInCallback,
	/// An exception was fired in JNI.
	javaException,
	/// Resource not found.
	resourceNotFound,
	/// An internal error has been fired.
	memoryError,
	/// An memory error has been fired.
	internalError,
	/// Error in the internal download process.
	downloadError,
	/// Invalid input format.
	invalidInputFormat,
	/// The LOTL is malformed.
	badInputLotl,
	/// The TSL does not conform to ETSI TS 119 612.
	badInputTsl,
	/// The manifest is malformed.
	badInputManifest,
	/// An unsupported Afnor XP Z42-101 identifier was found.
	unsupportedAfnorXpZ42_101Id,
	/// Decoding failed.
	decodingFailed,
	/// The BioSeal as expired.
	biosealExpired,
	/// Invalid certification chain.
	invalidCertificationChain,
	/// Invalid signature.
	invalidSignature,
	/// Authorized usage failed.
	authorizedUsageFailed,
	/// The signing certificate was expired when the BioSeal was generated.
	signingCertificateExpired,
	/// Invalid revocation info validity period.
	revocationInfoValidityPeriod,
	/// An unexpected error occurred during decryption..
	decryptionError,
	/// Language not found.
	languageNotAvailable,
	/// The BioSeal was not decoded.
	biosealNotDecoded,
	/// Function or request not supported.
	notSupported,
	/// Unable to verify the certificate chain.
	unableToVerifyCertificateChain,
	/// Need to verify the BioSeal.
	needToVerify,
	/// No signature found.
	noSignatureFound,
	/// This function is not implemented.
	functionNotImplemented,
	/// No certificate found.
	noCertificateFound,
	/// Internal SSL error.
	internalSslError,
	/// Unable to parse the given URL.
	urlParsingError,
	/// Document issuing country not supported.
	unsupportedDocumentIssuingCountry,
	/// Incorrect issuing agency code (IAC) in code prefix.
	wrongPrefix,
	/// TODO
	duplicateManifestField,
}

extension BiosealErrorX on BiosealError {
	/// Creates an enumeration value from int.
	/// param value The int value.
	/// return The created BiosealError.
  	static BiosealError fromValue(int value) {
    	switch (value) {
      	case 0:
        	return BiosealError.success;
      	case -1:
        	return BiosealError.insufficientBuffer;
      	case -2:
        	return BiosealError.objectDoesNotExist;
      	case -8000:
        	return BiosealError.base;
      	case -8999:
        	return BiosealError.minimum;
      	case -8001:
        	return BiosealError.invalidHandle;
      	case -8002:
        	return BiosealError.invalidParameter;
      	case -8003:
        	return BiosealError.exceptionInCallback;
      	case -8004:
        	return BiosealError.javaException;
      	case -8005:
        	return BiosealError.resourceNotFound;
      	case -8006:
        	return BiosealError.memoryError;
      	case -8007:
        	return BiosealError.internalError;
      	case -8008:
        	return BiosealError.downloadError;
      	case -8009:
        	return BiosealError.invalidInputFormat;
      	case -8010:
        	return BiosealError.badInputLotl;
      	case -8011:
        	return BiosealError.badInputTsl;
      	case -8012:
        	return BiosealError.badInputManifest;
      	case -8013:
        	return BiosealError.unsupportedAfnorXpZ42_101Id;
      	case -8014:
        	return BiosealError.decodingFailed;
      	case -8015:
        	return BiosealError.biosealExpired;
      	case -8016:
        	return BiosealError.invalidCertificationChain;
      	case -8017:
        	return BiosealError.invalidSignature;
      	case -8018:
        	return BiosealError.authorizedUsageFailed;
      	case -8019:
        	return BiosealError.signingCertificateExpired;
      	case -8020:
        	return BiosealError.revocationInfoValidityPeriod;
      	case -8021:
        	return BiosealError.decryptionError;
      	case -8022:
        	return BiosealError.languageNotAvailable;
      	case -8023:
        	return BiosealError.biosealNotDecoded;
      	case -8024:
        	return BiosealError.notSupported;
      	case -8025:
        	return BiosealError.unableToVerifyCertificateChain;
      	case -8026:
        	return BiosealError.needToVerify;
      	case -8027:
        	return BiosealError.noSignatureFound;
      	case -8028:
        	return BiosealError.functionNotImplemented;
      	case -8029:
        	return BiosealError.noCertificateFound;
      	case -8030:
        	return BiosealError.internalSslError;
      	case -8031:
        	return BiosealError.urlParsingError;
      	case -8032:
        	return BiosealError.unsupportedDocumentIssuingCountry;
      	case -8033:
        	return BiosealError.wrongPrefix;
      	case -8034:
        	return BiosealError.duplicateManifestField;
	    default:
        	throw Exception("Unable to find the specified enum value for $value");
    	}
  	}

	/// Gets the enumeration value as int.
	/// return The value.
  	int get value {
    	switch (this) {
      	case BiosealError.success:
        	return 0;
      	case BiosealError.insufficientBuffer:
        	return -1;
      	case BiosealError.objectDoesNotExist:
        	return -2;
      	case BiosealError.base:
        	return -8000;
      	case BiosealError.minimum:
        	return -8999;
      	case BiosealError.invalidHandle:
        	return -8001;
      	case BiosealError.invalidParameter:
        	return -8002;
      	case BiosealError.exceptionInCallback:
        	return -8003;
      	case BiosealError.javaException:
        	return -8004;
      	case BiosealError.resourceNotFound:
        	return -8005;
      	case BiosealError.memoryError:
        	return -8006;
      	case BiosealError.internalError:
        	return -8007;
      	case BiosealError.downloadError:
        	return -8008;
      	case BiosealError.invalidInputFormat:
        	return -8009;
      	case BiosealError.badInputLotl:
        	return -8010;
      	case BiosealError.badInputTsl:
        	return -8011;
      	case BiosealError.badInputManifest:
        	return -8012;
      	case BiosealError.unsupportedAfnorXpZ42_101Id:
        	return -8013;
      	case BiosealError.decodingFailed:
        	return -8014;
      	case BiosealError.biosealExpired:
        	return -8015;
      	case BiosealError.invalidCertificationChain:
        	return -8016;
      	case BiosealError.invalidSignature:
        	return -8017;
      	case BiosealError.authorizedUsageFailed:
        	return -8018;
      	case BiosealError.signingCertificateExpired:
        	return -8019;
      	case BiosealError.revocationInfoValidityPeriod:
        	return -8020;
      	case BiosealError.decryptionError:
        	return -8021;
      	case BiosealError.languageNotAvailable:
        	return -8022;
      	case BiosealError.biosealNotDecoded:
        	return -8023;
      	case BiosealError.notSupported:
        	return -8024;
      	case BiosealError.unableToVerifyCertificateChain:
        	return -8025;
      	case BiosealError.needToVerify:
        	return -8026;
      	case BiosealError.noSignatureFound:
        	return -8027;
      	case BiosealError.functionNotImplemented:
        	return -8028;
      	case BiosealError.noCertificateFound:
        	return -8029;
      	case BiosealError.internalSslError:
        	return -8030;
      	case BiosealError.urlParsingError:
        	return -8031;
      	case BiosealError.unsupportedDocumentIssuingCountry:
        	return -8032;
      	case BiosealError.wrongPrefix:
        	return -8033;
      	case BiosealError.duplicateManifestField:
        	return -8034;
		}
  	}
}
