//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------

/// Enumerates the BioSeal formats.
enum BiosealFormat {
	/// Undefined version.
	undefined,
	/// VDS compliant with the AFNOR standard XP Z42-101.
	vdsAfnorXpZ42_101,
	/// BioSeal compliant with the AFNOR XP Z42-105 standard and Otentik code specifications.
	vdsAfnorXpZ42_105,
	/// BioSeal compliant with the ISO 22376:2023 specifications.
	vdsIso22376_2023,
	/// BioSeal compliant with the ISO 22376:2023 specifications prefixed by the instance's ISO 15459 Issuing Agency Code.
	vdsIso22376WithIso15459Prefix,
	/// BioSeal compliant with the AFNOR XP Z42-105 standard prefixed by the instance's ISO 15459 Issuing Agency Code.
	vdsAfnorXpZ42105WithIso15459Prefix,
}

extension BiosealFormatX on BiosealFormat {
	/// Creates an enumeration value from int.
	/// param value The int value.
	/// return The created BiosealFormat.
  	static BiosealFormat fromValue(int value) {
    	switch (value) {
      	case 0:
        	return BiosealFormat.undefined;
      	case 1:
        	return BiosealFormat.vdsAfnorXpZ42_101;
      	case 2:
        	return BiosealFormat.vdsAfnorXpZ42_105;
      	case 3:
        	return BiosealFormat.vdsIso22376_2023;
      	case 4:
        	return BiosealFormat.vdsIso22376WithIso15459Prefix;
      	case 5:
        	return BiosealFormat.vdsAfnorXpZ42105WithIso15459Prefix;
	    default:
        	throw Exception("Unable to find the specified enum value for $value");
    	}
  	}

	/// Gets the enumeration value as int.
	/// return The value.
  	int get value {
    	switch (this) {
      	case BiosealFormat.undefined:
        	return 0;
      	case BiosealFormat.vdsAfnorXpZ42_101:
        	return 1;
      	case BiosealFormat.vdsAfnorXpZ42_105:
        	return 2;
      	case BiosealFormat.vdsIso22376_2023:
        	return 3;
      	case BiosealFormat.vdsIso22376WithIso15459Prefix:
        	return 4;
      	case BiosealFormat.vdsAfnorXpZ42105WithIso15459Prefix:
        	return 5;
		}
  	}
}
