//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------

/// Enumerates the error codes related to license management.
enum LicenseError {
	/// No error.
	success,
	/// The given buffer is too small.
	insufficientBuffer,
	/// The requested object does not exist; a 'null' should be returned if possible.
	objectDoesNotExist,
	/// Base error code of the License module.
	base,
	/// Minimum error code of the License module.
	minimum,
	/// The handle is invalid.
	invalidHandle,
	/// One of the parameters is incorrect.
	invalidParameter,
	/// No license was found.
	licenseNotFound,
	/// License is not valid or a required module is not licensed.
	invalidLicense,
	/// The license has expired.
	expiredLicense,
	/// Installation code does not match hardware ID.
	invalidHardwareId,
	/// The license is already fulfilled by another user (or computer).
	alreadyFulfilled,
	/// A required module is not licensed or the usage limit has been reached.
	moduleError,
	/// The license file cannot be accessed in write mode.
	writeError,
	/// The number of available licenses has been reached. There isn't any available license remaining.
	insufficientLicenseCredits,
	/// The serial is invalid.
	invalidSerial,
	/// Invalid login or password.
	invalidCredentials,
	/// The license is revoked or does not exist anymore.
	revokedLicense,
	/// The license is blocked after a predefined period.
	blockedLicense,
	/// The license server is currently unreachable.
	unreachableServer,
	/// An unexpected error happened.
	licenseOtherError,
	/// No license file was provided.
	missingFile,
	/// This device is not compatible with id3 licensing system.
	deviceNotCompatible,
	/// The activation of the license has failed.
	activationFailed,
	/// The license web services are unavailable.
	serviceUnavailable,
	/// The license is not renewable.
	notRenewable,
	/// The digest is invalid. The received file may be corrupted.
	digestInvalid,
	/// No credit available for this product.
	productUnavailable,
	/// You are not allowed to do this.
	unauthorizedAction,
	/// The license will be shortly blocked.
	preBlockedLicense,
}

extension LicenseErrorX on LicenseError {
	/// Creates an enumeration value from int.
	/// param value The int value.
	/// return The created LicenseError.
  	static LicenseError fromValue(int value) {
    	switch (value) {
      	case 0:
        	return LicenseError.success;
      	case -1:
        	return LicenseError.insufficientBuffer;
      	case -2:
        	return LicenseError.objectDoesNotExist;
      	case -100:
        	return LicenseError.base;
      	case -999:
        	return LicenseError.minimum;
      	case -101:
        	return LicenseError.invalidHandle;
      	case -102:
        	return LicenseError.invalidParameter;
      	case -103:
        	return LicenseError.licenseNotFound;
      	case -104:
        	return LicenseError.invalidLicense;
      	case -105:
        	return LicenseError.expiredLicense;
      	case -106:
        	return LicenseError.invalidHardwareId;
      	case -107:
        	return LicenseError.alreadyFulfilled;
      	case -108:
        	return LicenseError.moduleError;
      	case -109:
        	return LicenseError.writeError;
      	case -110:
        	return LicenseError.insufficientLicenseCredits;
      	case -111:
        	return LicenseError.invalidSerial;
      	case -112:
        	return LicenseError.invalidCredentials;
      	case -113:
        	return LicenseError.revokedLicense;
      	case -114:
        	return LicenseError.blockedLicense;
      	case -115:
        	return LicenseError.unreachableServer;
      	case -116:
        	return LicenseError.licenseOtherError;
      	case -117:
        	return LicenseError.missingFile;
      	case -118:
        	return LicenseError.deviceNotCompatible;
      	case -119:
        	return LicenseError.activationFailed;
      	case -120:
        	return LicenseError.serviceUnavailable;
      	case -121:
        	return LicenseError.notRenewable;
      	case -122:
        	return LicenseError.digestInvalid;
      	case -123:
        	return LicenseError.productUnavailable;
      	case -124:
        	return LicenseError.unauthorizedAction;
      	case -125:
        	return LicenseError.preBlockedLicense;
	    default:
        	throw Exception("Unable to find the specified enum value for $value");
    	}
  	}

	/// Gets the enumeration value as int.
	/// return The value.
  	int get value {
    	switch (this) {
      	case LicenseError.success:
        	return 0;
      	case LicenseError.insufficientBuffer:
        	return -1;
      	case LicenseError.objectDoesNotExist:
        	return -2;
      	case LicenseError.base:
        	return -100;
      	case LicenseError.minimum:
        	return -999;
      	case LicenseError.invalidHandle:
        	return -101;
      	case LicenseError.invalidParameter:
        	return -102;
      	case LicenseError.licenseNotFound:
        	return -103;
      	case LicenseError.invalidLicense:
        	return -104;
      	case LicenseError.expiredLicense:
        	return -105;
      	case LicenseError.invalidHardwareId:
        	return -106;
      	case LicenseError.alreadyFulfilled:
        	return -107;
      	case LicenseError.moduleError:
        	return -108;
      	case LicenseError.writeError:
        	return -109;
      	case LicenseError.insufficientLicenseCredits:
        	return -110;
      	case LicenseError.invalidSerial:
        	return -111;
      	case LicenseError.invalidCredentials:
        	return -112;
      	case LicenseError.revokedLicense:
        	return -113;
      	case LicenseError.blockedLicense:
        	return -114;
      	case LicenseError.unreachableServer:
        	return -115;
      	case LicenseError.licenseOtherError:
        	return -116;
      	case LicenseError.missingFile:
        	return -117;
      	case LicenseError.deviceNotCompatible:
        	return -118;
      	case LicenseError.activationFailed:
        	return -119;
      	case LicenseError.serviceUnavailable:
        	return -120;
      	case LicenseError.notRenewable:
        	return -121;
      	case LicenseError.digestInvalid:
        	return -122;
      	case LicenseError.productUnavailable:
        	return -123;
      	case LicenseError.unauthorizedAction:
        	return -124;
      	case LicenseError.preBlockedLicense:
        	return -125;
		}
  	}
}
