//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
import 'dart:ffi';
import 'package:ffi/ffi.dart';
import 'bioseal_sdk_generated_bindings.dart';

/// Contains data for VerificationResult
class VerificationResult {
  final Pointer<id3BiosealVerificationResult> _handle;

  static final Finalizer<Pointer<id3BiosealVerificationResult>> _finalizer = Finalizer((handle) => calloc.free(handle));

  Pointer<id3BiosealVerificationResult> get handle => _handle;

  id3BiosealVerificationResult get struct => _handle.ref;

  /// Initializes a new instance of the VerificationResult class from its members.
  VerificationResult(this._handle);

  /// Create a new instance of the VerificationResult.
  VerificationResult.create() : _handle = calloc() {
    _finalizer.attach(this, _handle, detach: this);
  }

  /// Create a new instance of the VerificationResult from the native struct
  VerificationResult.createFromStruct(id3BiosealVerificationResult value) : _handle = calloc() {
    _handle.ref.CertificationChainVerified = value.CertificationChainVerified;
    _handle.ref.GovernanceValid = value.GovernanceValid;
    _handle.ref.LotlGovernanceValid = value.LotlGovernanceValid;
    _handle.ref.TslGovernanceValid = value.TslGovernanceValid;
    _handle.ref.ManifestGovernanceValid = value.ManifestGovernanceValid;
    _handle.ref.VdsNotExpired = value.VdsNotExpired;
    _handle.ref.VdsSignatureVerified = value.VdsSignatureVerified;
    _handle.ref.CaCertificateVerified = value.CaCertificateVerified;
    _handle.ref.SigningCertificateUsageAuthorized = value.SigningCertificateUsageAuthorized;
    _finalizer.attach(this, _handle, detach: this);
  }

  void dispose() {
    calloc.free(_handle);
    _finalizer.detach(this);
  }

  // Accessors
  
  /// 
  /// Indicates whether the certification chain has been verified.
  /// 
  /// throws BiosealException An error has occurred during Bioseal Library execution.
  int get certificationChainVerified => _handle.ref.CertificationChainVerified;
  set certificationChainVerified(int value) => _handle.ref.CertificationChainVerified = value;

  /// 
  /// Indicates whether the governance that issued the VDS is valid.
  /// 
  /// throws BiosealException An error has occurred during Bioseal Library execution.
  int get governanceValid => _handle.ref.GovernanceValid;
  set governanceValid(int value) => _handle.ref.GovernanceValid = value;

  /// 
  /// Indicates whether the LoTL governance that issued the VDS is valid.
  /// 
  /// throws BiosealException An error has occurred during Bioseal Library execution.
  int get lotlGovernanceValid => _handle.ref.LotlGovernanceValid;
  set lotlGovernanceValid(int value) => _handle.ref.LotlGovernanceValid = value;

  /// 
  /// Indicates whether the TSL governance that issued the VDS is valid.
  /// 
  /// throws BiosealException An error has occurred during Bioseal Library execution.
  int get tslGovernanceValid => _handle.ref.TslGovernanceValid;
  set tslGovernanceValid(int value) => _handle.ref.TslGovernanceValid = value;

  /// 
  /// Indicates whether the manifest governance that issued the VDS is valid.
  /// 
  /// throws BiosealException An error has occurred during Bioseal Library execution.
  int get manifestGovernanceValid => _handle.ref.ManifestGovernanceValid;
  set manifestGovernanceValid(int value) => _handle.ref.ManifestGovernanceValid = value;

  /// 
  /// Indicates whether the VDS is still valid, ie. it has not expired.
  /// 
  /// throws BiosealException An error has occurred during Bioseal Library execution.
  int get vdsNotExpired => _handle.ref.VdsNotExpired;
  set vdsNotExpired(int value) => _handle.ref.VdsNotExpired = value;

  /// 
  /// Indicates whether the signature of the VDS has been verified.
  /// 
  /// throws BiosealException An error has occurred during Bioseal Library execution.
  int get vdsSignatureVerified => _handle.ref.VdsSignatureVerified;
  set vdsSignatureVerified(int value) => _handle.ref.VdsSignatureVerified = value;

  /// 
  /// Indicates whether the CA certificate has been verified.
  /// 
  /// throws BiosealException An error has occurred during Bioseal Library execution.
  int get caCertificateVerified => _handle.ref.CaCertificateVerified;
  set caCertificateVerified(int value) => _handle.ref.CaCertificateVerified = value;

  /// 
  /// Indicates whether the signing certificate was legitimately authorized to sign the VDS.
  /// 
  /// throws BiosealException An error has occurred during Bioseal Library execution.
  int get signingCertificateUsageAuthorized => _handle.ref.SigningCertificateUsageAuthorized;
  set signingCertificateUsageAuthorized(int value) => _handle.ref.SigningCertificateUsageAuthorized = value;


  Map<String, dynamic> toMap() {
    return {
        'certificationChainVerified': certificationChainVerified,
        'governanceValid': governanceValid,
        'lotlGovernanceValid': lotlGovernanceValid,
        'tslGovernanceValid': tslGovernanceValid,
        'manifestGovernanceValid': manifestGovernanceValid,
        'vdsNotExpired': vdsNotExpired,
        'vdsSignatureVerified': vdsSignatureVerified,
        'caCertificateVerified': caCertificateVerified,
        'signingCertificateUsageAuthorized': signingCertificateUsageAuthorized,
    };
  }
}
