//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------

/// Enumerates the visibility conditions.
/// Note: The optional visibility condition extension indicates under which condition the presentation view or certain field in the presentation view should be displayed.
enum VisibilityCondition {
	/// The presentation view or the specified field shall always be displayed. This is the default value.
	always,
	/// The BioSeal signature must be valid.
	signatureValidated,
	/// The signature must be valid and the bioseal must not be expired.
	notExpired,
	/// The above conditions must be met, and user authentication must be successful.
	userAuthenticated,
	///  The presentation view or the specified field must never be displayed. In this case, the application must not inject the data into the presentation view.
	never,
}

extension VisibilityConditionX on VisibilityCondition {
	/// Creates an enumeration value from int.
	/// param value The int value.
	/// return The created VisibilityCondition.
  	static VisibilityCondition fromValue(int value) {
    	switch (value) {
      	case 0:
        	return VisibilityCondition.always;
      	case 1:
        	return VisibilityCondition.signatureValidated;
      	case 2:
        	return VisibilityCondition.notExpired;
      	case 3:
        	return VisibilityCondition.userAuthenticated;
      	case 4:
        	return VisibilityCondition.never;
	    default:
        	throw Exception("Unable to find the specified enum value for $value");
    	}
  	}

	/// Gets the enumeration value as int.
	/// return The value.
  	int get value {
    	switch (this) {
      	case VisibilityCondition.always:
        	return 0;
      	case VisibilityCondition.signatureValidated:
        	return 1;
      	case VisibilityCondition.notExpired:
        	return 2;
      	case VisibilityCondition.userAuthenticated:
        	return 3;
      	case VisibilityCondition.never:
        	return 4;
		}
  	}
}
