//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3Bioseal_h
#define _id3Bioseal_h

#include <id3Bioseal/id3BiosealExports.h>
#include <id3Bioseal/id3BiosealFormat.h>
#include <id3Bioseal/id3BiosealDateTime.h>
#include <id3Bioseal/id3BiosealField.h>
#include <id3Bioseal/id3BiosealManifest.h>
#include <id3Bioseal/id3BiosealStringArray.h>
#include <id3Bioseal/id3BiosealResourceCallbackHandler.h>
#include <id3Bioseal/id3BiosealCertificateInformation.h>
#include <id3Bioseal/id3BiosealBiometricVerificationRules.h>
#include <id3Bioseal/id3BiosealLog.h>
#include <id3Bioseal/id3BiosealUserAuthentication.h>
#include <id3Bioseal/id3BiosealDecryptionCallbackHandler.h>
#include <id3Bioseal/id3BiosealVerificationResult.h>
#include <id3Bioseal/id3BiosealFieldExtensionType.h>
#include <id3Bioseal/id3BiosealFieldList.h>
#include <id3Bioseal/id3BiosealBiometricDataType.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a BioSeal code.
 */

/**
 * Handle to a Bioseal object of the Bioseal module.
 */
typedef struct id3Bioseal *ID3_BIOSEAL;

/**
 * @brief Initializes a Bioseal object.
 * @param phBioseal [in] Handle to the Bioseal object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_Initialize(ID3_BIOSEAL *phBioseal);

/**
 * @brief Releases the memory allocated for a Bioseal.
 * @param phBioseal [in] Handle to the Bioseal object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_Dispose(ID3_BIOSEAL *phBioseal);

/**
 * @brief Checks if the handle object is valid.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_CheckHandle(ID3_BIOSEAL hBioseal);

/**
 * Gets the the BioSeal format without prefix information.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param eFormat [out] The BioSeal format without prefix information.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetFormat(ID3_BIOSEAL hBioseal, id3BiosealFormat *eFormat);

/**
 * Gets the the BioSeal format with prefix information.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param eExtendedFormat [out] The BioSeal format with prefix information.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetExtendedFormat(ID3_BIOSEAL hBioseal, id3BiosealFormat *eExtendedFormat);

/**
 * Gets the the document issuing date.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param hDocumentIssuingDate [out] The document issuing date.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetDocumentIssuingDate(ID3_BIOSEAL hBioseal, ID3_BIOSEAL_DATE_TIME hDocumentIssuingDate);

/**
 * Gets the the BioSeal signature date.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param hSignatureDate [out] The BioSeal signature date.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetSignatureDate(ID3_BIOSEAL hBioseal, ID3_BIOSEAL_DATE_TIME hSignatureDate);

/**
 * Gets the the document type identification code. For AFNOR XP Z42-101 only.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param documentType [out] The document type identification code. For AFNOR XP Z42-101 only.
 * @param documentTypeSize [in,out] Size of the 'documentType' buffer. Receives the minimum required buffer size to hold the 'documentType' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'documentType' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetDocumentType(ID3_BIOSEAL hBioseal, char *documentType, int *documentTypeSize);

/**
 * Gets the the identification code of the scope on which the document type is defined. For AFNOR XP Z42-101 only.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param documentScope [out] The identification code of the scope on which the document type is defined. For AFNOR XP Z42-101 only.
 * @param documentScopeSize [in,out] Size of the 'documentScope' buffer. Receives the minimum required buffer size to hold the 'documentScope' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'documentScope' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetDocumentScope(ID3_BIOSEAL hBioseal, char *documentScope, int *documentScopeSize);

/**
 * Gets the the identifier of the country issuing the document. For AFNOR XP Z42-101 only.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param documentIssuingCountry [out] The identifier of the country issuing the document. For AFNOR XP Z42-101 only.
 * @param documentIssuingCountrySize [in,out] Size of the 'documentIssuingCountry' buffer. Receives the minimum required buffer size to hold the 'documentIssuingCountry' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'documentIssuingCountry' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetDocumentIssuingCountry(ID3_BIOSEAL hBioseal, char *documentIssuingCountry, int *documentIssuingCountrySize);

/**
 * Gets the the manifest ID in ISO 22385 format.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param manifestId [out] The manifest ID in ISO 22385 format.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetManifestId(ID3_BIOSEAL hBioseal, int *manifestId);

/**
 * Gets the the payload containing the data specific to the BioSeal.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param hPayload [out] The payload containing the data specific to the BioSeal.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetPayload(ID3_BIOSEAL hBioseal, ID3_BIOSEAL_FIELD hPayload);

/**
 * Gets the the manifest information.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param hManifest [out] The manifest information.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetManifest(ID3_BIOSEAL hBioseal, ID3_BIOSEAL_MANIFEST hManifest);

/**
 * Gets the indicates whether the BioSeal contains images.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param containsImages [out] Indicates whether the BioSeal contains images.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetContainsImages(ID3_BIOSEAL hBioseal, bool *containsImages);

/**
 * Gets the indicates whether the BioSeal contains portraits.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param containsPortraits [out] Indicates whether the BioSeal contains portraits.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetContainsPortraits(ID3_BIOSEAL hBioseal, bool *containsPortraits);

/**
 * Gets the indicates whether the BioSeal contains biometric data.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param containsBiometrics [out] Indicates whether the BioSeal contains biometric data.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetContainsBiometrics(ID3_BIOSEAL hBioseal, bool *containsBiometrics);

/**
 * Gets the indicates whether the BioSeal contains face templates.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param containsFaceTemplates [out] Indicates whether the BioSeal contains face templates.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetContainsFaceTemplates(ID3_BIOSEAL hBioseal, bool *containsFaceTemplates);

/**
 * Gets the indicates whether the BioSeal contains finger templates.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param containsFingerTemplates [out] Indicates whether the BioSeal contains finger templates.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetContainsFingerTemplates(ID3_BIOSEAL hBioseal, bool *containsFingerTemplates);

/**
 * Gets the indicates whether the BioSeal contains extensions.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param containsExtensions [out] Indicates whether the BioSeal contains extensions.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetContainsExtensions(ID3_BIOSEAL hBioseal, bool *containsExtensions);

/**
 * Gets the the list of supported languages declared in the manifest.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param hSupportedLanguages [out] The list of supported languages declared in the manifest.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetSupportedLanguages(ID3_BIOSEAL hBioseal, ID3_BIOSEAL_STRING_ARRAY hSupportedLanguages);

/**
 * Gets the the presentation view, in HTML format, created from the manifest and the data contained in the BioSeal.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param htmlView [out] The presentation view, in HTML format, created from the manifest and the data contained in the BioSeal.
 * @param htmlViewSize [in,out] Size of the 'htmlView' buffer. Receives the minimum required buffer size to hold the 'htmlView' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'htmlView' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetHtmlView(ID3_BIOSEAL hBioseal, char *htmlView, int *htmlViewSize);

/**
 * Sets the the callback used to retrieve an external resource (LoTL, TSL, certificate, manifest, CRL) with specified URI.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param externalResourceCallback [in] The callback used to retrieve an external resource (LoTL, TSL, certificate, manifest, CRL) with specified URI.
 * @param context [in] The callback context.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_SetExternalResourceCallback(ID3_BIOSEAL hBioseal, id3BiosealResourceCallbackHandler externalResourceCallback, void *context);

/**
 * Gets the the issuing reference of the certificate authority.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param certificateAuthorityReference [out] The issuing reference of the certificate authority.
 * @param certificateAuthorityReferenceSize [in,out] Size of the 'certificateAuthorityReference' buffer. Receives the minimum required buffer size to hold the 'certificateAuthorityReference' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'certificateAuthorityReference' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetCertificateAuthorityReference(ID3_BIOSEAL hBioseal, char *certificateAuthorityReference, int *certificateAuthorityReferenceSize);

/**
 * Gets the the identifier of the certificate.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param certificateIdentifier [out] The identifier of the certificate.
 * @param certificateIdentifierSize [in,out] Size of the 'certificateIdentifier' buffer. Receives the minimum required buffer size to hold the 'certificateIdentifier' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'certificateIdentifier' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetCertificateIdentifier(ID3_BIOSEAL hBioseal, char *certificateIdentifier, int *certificateIdentifierSize);

/**
 * Gets the the detailed certificate information.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param hCertificateInformation [out] The detailed certificate information.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetCertificateInformation(ID3_BIOSEAL hBioseal, ID3_BIOSEAL_CERTIFICATE_INFORMATION hCertificateInformation);

/**
 * Gets the the BioSeal expiration date.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param hExpirationDate [out] The BioSeal expiration date.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetExpirationDate(ID3_BIOSEAL hBioseal, ID3_BIOSEAL_DATE_TIME hExpirationDate);

/**
 * Gets the indicates whether the BioSeal contains auxiliary data.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param containsAuxData [out] Indicates whether the BioSeal contains auxiliary data.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetContainsAuxData(ID3_BIOSEAL hBioseal, bool *containsAuxData);

/**
 * Gets the the auxiliary data containing additional information specific to the BioSeal.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param hAuxData [out] The auxiliary data containing additional information specific to the BioSeal.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetAuxData(ID3_BIOSEAL hBioseal, ID3_BIOSEAL_FIELD hAuxData);

/**
 * Gets the the list of biometric verification rules.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param hBiometricVerificationRules [out] The list of biometric verification rules.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetBiometricVerificationRules(ID3_BIOSEAL hBioseal, ID3_BIOSEAL_BIOMETRIC_VERIFICATION_RULES hBiometricVerificationRules);

/**
 * Gets the the log.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param hLog [out] The log.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetLog(ID3_BIOSEAL hBioseal, ID3_BIOSEAL_LOG hLog);

/**
 * Gets the the user authentication parameters.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param hUserAuthentication [out] The user authentication parameters.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetUserAuthentication(ID3_BIOSEAL hBioseal, ID3_BIOSEAL_USER_AUTHENTICATION hUserAuthentication);

/**
 * Gets the the header version.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param headerVersion [out] The header version.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetHeaderVersion(ID3_BIOSEAL hBioseal, int *headerVersion);

/**
 * Sets the the callback used to decrypt a field.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param decryptionCallback [in] The callback used to decrypt a field.
 * @param context [in] The callback context.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_SetDecryptionCallback(ID3_BIOSEAL hBioseal, id3BiosealDecryptionCallbackHandler decryptionCallback, void *context);

/**
 * Gets the indicates whether the governance certification chain is verified.
 * Hint: Default value is true.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param verifiesGovernance [out] Indicates whether the governance certification chain is verified.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetVerifiesGovernance(ID3_BIOSEAL hBioseal, bool *verifiesGovernance);

/**
 * Sets the indicates whether the governance certification chain is verified.
 * Hint: Default value is true.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param verifiesGovernance [in] Indicates whether the governance certification chain is verified.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_SetVerifiesGovernance(ID3_BIOSEAL hBioseal, bool verifiesGovernance);

/**
 * Gets the indicates whether the VDS signature is verified.
 * Hint: Default value is true.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param verifiesSignature [out] Indicates whether the VDS signature is verified.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetVerifiesSignature(ID3_BIOSEAL hBioseal, bool *verifiesSignature);

/**
 * Sets the indicates whether the VDS signature is verified.
 * Hint: Default value is true.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param verifiesSignature [in] Indicates whether the VDS signature is verified.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_SetVerifiesSignature(ID3_BIOSEAL hBioseal, bool verifiesSignature);

/**
 * Gets the indicates whether the verification process should verify the next CRL update.
 * Hint: Default value is true.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param verifiesCrlUpdates [out] Indicates whether the verification process should verify the next CRL update.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetVerifiesCrlUpdates(ID3_BIOSEAL hBioseal, bool *verifiesCrlUpdates);

/**
 * Sets the indicates whether the verification process should verify the next CRL update.
 * Hint: Default value is true.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param verifiesCrlUpdates [in] Indicates whether the verification process should verify the next CRL update.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_SetVerifiesCrlUpdates(ID3_BIOSEAL hBioseal, bool verifiesCrlUpdates);

/**
 * Gets the the last verification result.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param sVerificationResult [out] The last verification result.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetVerificationResult(ID3_BIOSEAL hBioseal, id3BiosealVerificationResult *sVerificationResult);

/**
 * Gets the the URL to the LoTL (List of Trust List).
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param lotlUrl [out] The URL to the LoTL (List of Trust List).
 * @param lotlUrlSize [in,out] Size of the 'lotlUrl' buffer. Receives the minimum required buffer size to hold the 'lotlUrl' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'lotlUrl' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetLotlUrl(ID3_BIOSEAL hBioseal, char *lotlUrl, int *lotlUrlSize);

/**
 * Sets the the URL to the LoTL (List of Trust List).
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param lotlUrl [in] The URL to the LoTL (List of Trust List).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_SetLotlUrl(ID3_BIOSEAL hBioseal, const char *lotlUrl);

/**
 * Gets the the URL to the TSL (Trust Service List).
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param tslUrl [out] The URL to the TSL (Trust Service List).
 * @param tslUrlSize [in,out] Size of the 'tslUrl' buffer. Receives the minimum required buffer size to hold the 'tslUrl' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'tslUrl' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetTslUrl(ID3_BIOSEAL hBioseal, char *tslUrl, int *tslUrlSize);

/**
 * Sets the the URL to the TSL (Trust Service List).
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param tslUrl [in] The URL to the TSL (Trust Service List).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_SetTslUrl(ID3_BIOSEAL hBioseal, const char *tslUrl);

/**
 * Gets the the URL to the manifest.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param manifestUrl [out] The URL to the manifest.
 * @param manifestUrlSize [in,out] Size of the 'manifestUrl' buffer. Receives the minimum required buffer size to hold the 'manifestUrl' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'manifestUrl' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetManifestUrl(ID3_BIOSEAL hBioseal, char *manifestUrl, int *manifestUrlSize);

/**
 * Sets the the URL to the manifest.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param manifestUrl [in] The URL to the manifest.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_SetManifestUrl(ID3_BIOSEAL hBioseal, const char *manifestUrl);

/**
 * Gets the the Issuing Agency Code (IAC). For ISO 22376:2023 format only.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param issuingAgencyCode [out] The Issuing Agency Code (IAC). For ISO 22376:2023 format only.
 * @param issuingAgencyCodeSize [in,out] Size of the 'issuingAgencyCode' buffer. Receives the minimum required buffer size to hold the 'issuingAgencyCode' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'issuingAgencyCode' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetIssuingAgencyCode(ID3_BIOSEAL hBioseal, char *issuingAgencyCode, int *issuingAgencyCodeSize);

/**
 * Gets the prefix is present.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param hasPrefix [out] Prefix is present.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetHasPrefix(ID3_BIOSEAL hBioseal, bool *hasPrefix);

/**
 * Gets the the prefix.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param prefix [out] The prefix.
 * @param prefixSize [in,out] Size of the 'prefix' buffer. Receives the minimum required buffer size to hold the 'prefix' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'prefix' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetPrefix(ID3_BIOSEAL hBioseal, char *prefix, int *prefixSize);

/**
 * Gets the the data of the VDS as a binary array.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param data [out] The data of the VDS as a binary array.
 * @param dataSize [in,out] Size of the 'data' buffer. Receives the number of elements in the 'data' array.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetData(ID3_BIOSEAL hBioseal, unsigned char *data, int *dataSize);

/**
 * Clears all internal data.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_Clear(ID3_BIOSEAL hBioseal);

/**
 * Decodes and verifies the specified data buffer.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param data [in] The data buffer containing the BioSeal code.
 * @param dataSize [in] Size of the 'data' buffer.
 * @param sResult [out] Receives the verification results.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_VerifyFromBuffer(ID3_BIOSEAL hBioseal, unsigned char *data, int dataSize, id3BiosealVerificationResult *sResult);

/**
 * Decodes and verifies the specified string buffer.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param data [in] The string buffer containing the BioSeal code.
 * @param sResult [out] Receives the verification results.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_VerifyFromString(ID3_BIOSEAL hBioseal, const char *data, id3BiosealVerificationResult *sResult);

/**
 * Builds the presentation view, in HTML format.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param language [in] The expected language.
 * @param userAuthenticated [in] A value indicating whether the user is authenticated.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_BuildHtmlView(ID3_BIOSEAL hBioseal, const char *language, bool userAuthenticated);

/**
 * Gets the name of the document.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param language [in] The expected language.
 * @param documentName [out] The name of the document.
 * @param documentNameSize [in,out] Size of the 'documentName' buffer. Receives the minimum required buffer size to hold the 'documentName' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'documentName' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetDocumentName(ID3_BIOSEAL hBioseal, const char *language, char *documentName, int *documentNameSize);

/**
 * Gets the description of the document.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param language [in] The expected language.
 * @param documentDescription [out] The description of the document.
 * @param documentDescriptionSize [in,out] Size of the 'documentDescription' buffer. Receives the minimum required buffer size to hold the 'documentDescription' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'documentDescription' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_GetDocumentDescription(ID3_BIOSEAL hBioseal, const char *language, char *documentDescription, int *documentDescriptionSize);

/**
 * Finds fields by extension type.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param eFieldExtensionType [in] The field extension type.
 * @param hResultFieldList [out] The list of fields with the extension.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_FindFieldsByExtension(ID3_BIOSEAL hBioseal, id3BiosealFieldExtensionType eFieldExtensionType, ID3_BIOSEAL_FIELD_LIST hResultFieldList);

/**
 * Finds fields by extension name.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param extensionName [in] A string that identifies the extension name.
 * @param hResultFieldList [out] The list of fields with the extension.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_FindFieldsByExtensionName(ID3_BIOSEAL hBioseal, const char *extensionName, ID3_BIOSEAL_FIELD_LIST hResultFieldList);

/**
 * Finds the biometric field in payload.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param eBiometricDataType [in] The type of biometric data.
 * @param biometricFormat [in] The format of the biometric data.
 * @param hResultFieldList [out] The list of fields containing biometric data.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_FindBiometrics(ID3_BIOSEAL hBioseal, id3BiosealBiometricDataType eBiometricDataType, const char *biometricFormat, ID3_BIOSEAL_FIELD_LIST hResultFieldList);

/**
 * Builds a representation of the VDS, in JSON format.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param indentation [in] The indentation of the json string.
 * @param result [out] The result.
 * @param resultSize [in,out] Size of the 'result' buffer. Receives the minimum required buffer size to hold the 'result' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'result' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_BuildVdsAsJson(ID3_BIOSEAL hBioseal, const char *indentation, char *result, int *resultSize);

/**
 * Builds a representation of the VDS, in a XML format.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param language [in] The expected language.
 * @param result [out] The result.
 * @param resultSize [in,out] Size of the 'result' buffer. Receives the minimum required buffer size to hold the 'result' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'result' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_BuildVdsAsXml(ID3_BIOSEAL hBioseal, const char *language, char *result, int *resultSize);

/**
 * Clears the internal download cache.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_ClearDownloadCache(ID3_BIOSEAL hBioseal);

/**
 * Sets the time zone offset in seconds.
 * @param hBioseal [in] Handle to the Bioseal object.
 * @param offsetInSeconds [in] The time zone offset in seconds.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_SetTimeZoneOffset(ID3_BIOSEAL hBioseal, int offsetInSeconds);

/**
 * Computes SHA256.
 * @param data [in] Input data.
 * @param dataSize [in] Size of the 'data' buffer.
 * @param result [out] The result.
 * @param resultSize [in,out] Size of the 'result' buffer. Receives the minimum required buffer size to hold the 'result' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'result' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3Bioseal_ComputeSha256(unsigned char *data, int dataSize, char *result, int *resultSize);

#ifdef __cplusplus
}
#endif

#endif
