//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3BiosealExtensionList_h
#define _id3BiosealExtensionList_h

#include <id3Bioseal/id3BiosealExports.h>
#include <id3Bioseal/id3BiosealStringArray.h>
#include <id3Bioseal/id3BiosealExtension.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a list of extensions.
 */

/**
 * Handle to a ExtensionList object of the Bioseal module.
 */
typedef struct id3BiosealExtensionList *ID3_BIOSEAL_EXTENSION_LIST;

/**
 * @brief Initializes a ExtensionList object.
 * @param phExtensionList [in] Handle to the ExtensionList object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtensionList_Initialize(ID3_BIOSEAL_EXTENSION_LIST *phExtensionList);

/**
 * @brief Releases the memory allocated for a ExtensionList.
 * @param phExtensionList [in] Handle to the ExtensionList object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtensionList_Dispose(ID3_BIOSEAL_EXTENSION_LIST *phExtensionList);

/**
 * @brief Checks if the handle object is valid.
 * @param hExtensionList [in] Handle to the ExtensionList object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtensionList_CheckHandle(ID3_BIOSEAL_EXTENSION_LIST hExtensionList);

/**
 * Gets the gets a string list containing the keys in the dict.
 * @param hExtensionList [in] Handle to the ExtensionList object.
 * @param hKeys [out] Gets a string list containing the keys in the dict.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtensionList_GetKeys(ID3_BIOSEAL_EXTENSION_LIST hExtensionList, ID3_BIOSEAL_STRING_ARRAY hKeys);

/**
 * Gets an item of the ExtensionList object.
 * @param hExtensionList [in] Handle to the ExtensionList object.
 * @param key [in] Unique key of the Extension item to get.
 * @param hExtensionItem [out] Extension item to get.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtensionList_Get(ID3_BIOSEAL_EXTENSION_LIST hExtensionList, const char *key, ID3_BIOSEAL_EXTENSION hExtensionItem);

/**
 * Gets the number of elements in the ExtensionList object.
 * @param hExtensionList [in] Handle to the ExtensionList object.
 * @param count [out] Number of elements in the Extension object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtensionList_GetCount(ID3_BIOSEAL_EXTENSION_LIST hExtensionList, int *count);

/**
 * Determines whether the ExtensionListobject contains the specified key.
 * @param hExtensionList [in] Handle to the ExtensionList object.
 * @param key [in] The key to locate in the Extension object.
 * @param result [out] true if the Extension object contains an element with the specified key, otherwise false.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtensionList_ContainsKey(ID3_BIOSEAL_EXTENSION_LIST hExtensionList, const char *key, bool *result);

/**
 * Clear the list.
 * @param hExtensionList [in] Handle to the ExtensionList object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealExtensionList_Clear(ID3_BIOSEAL_EXTENSION_LIST hExtensionList);

#ifdef __cplusplus
}
#endif

#endif
