//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3BiosealLog_h
#define _id3BiosealLog_h

#include <id3Bioseal/id3BiosealExports.h>
#include <id3Bioseal/id3BiosealLogItem.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents the log.
 */

/**
 * Handle to a Log object of the Bioseal module.
 */
typedef struct id3BiosealLog *ID3_BIOSEAL_LOG;

/**
 * @brief Initializes a Log object.
 * @param phLog [in] Handle to the Log object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLog_Initialize(ID3_BIOSEAL_LOG *phLog);

/**
 * @brief Releases the memory allocated for a Log.
 * @param phLog [in] Handle to the Log object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLog_Dispose(ID3_BIOSEAL_LOG *phLog);

/**
 * @brief Checks if the handle object is valid.
 * @param hLog [in] Handle to the Log object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLog_CheckHandle(ID3_BIOSEAL_LOG hLog);

/**
 * Gets an item of the Log object.
 * @param hLog [in] Handle to the Log object.
 * @param index [in] Index of the LogItem item to get.
 * @param hLogItemItem [out] LogItem item to get.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLog_Get(ID3_BIOSEAL_LOG hLog, int index, ID3_BIOSEAL_LOG_ITEM hLogItemItem);

/**
 * Gets the number of elements in the Log object.
 * @param hLog [in] Handle to the Log object.
 * @param count [out] Number of elements in the LogItem object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLog_GetCount(ID3_BIOSEAL_LOG hLog, int *count);

/**
 * Exports a representation of the log, in JSON format.
 * @param hLog [in] Handle to the Log object.
 * @param indentation [in] The indentation of the json string.
 * @param debug [in] Indicates whether debug information is added to the log.
 * @param result [out] The result.
 * @param resultSize [in,out] Size of the 'result' buffer. Receives the minimum required buffer size to hold the 'result' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'result' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLog_ExportToJson(ID3_BIOSEAL_LOG hLog, const char *indentation, bool debug, char *result, int *resultSize);

/**
 * Exports a representation of the log, in HTML format.
 * @param hLog [in] Handle to the Log object.
 * @param debug [in] Indicates whether debug information is added to the log.
 * @param result [out] The result.
 * @param resultSize [in,out] Size of the 'result' buffer. Receives the minimum required buffer size to hold the 'result' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'result' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLog_ExportToHtml(ID3_BIOSEAL_LOG hLog, bool debug, char *result, int *resultSize);

#ifdef __cplusplus
}
#endif

#endif
