//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3BiosealLotl_h
#define _id3BiosealLotl_h

#include <id3Bioseal/id3BiosealExports.h>
#include <id3Bioseal/id3BiosealLog.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Provides information about the LoTL (List of Trust List).
 */

/**
 * Handle to a Lotl object of the Bioseal module.
 */
typedef struct id3BiosealLotl *ID3_BIOSEAL_LOTL;

/**
 * @brief Initializes a Lotl object.
 * @param phLotl [in] Handle to the Lotl object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLotl_Initialize(ID3_BIOSEAL_LOTL *phLotl);

/**
 * @brief Releases the memory allocated for a Lotl.
 * @param phLotl [in] Handle to the Lotl object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLotl_Dispose(ID3_BIOSEAL_LOTL *phLotl);

/**
 * @brief Checks if the handle object is valid.
 * @param hLotl [in] Handle to the Lotl object.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLotl_CheckHandle(ID3_BIOSEAL_LOTL hLotl);

/**
 * Gets the get the log.
 * @param hLotl [in] Handle to the Lotl object.
 * @param hLog [out] Get the log.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLotl_GetLog(ID3_BIOSEAL_LOTL hLotl, ID3_BIOSEAL_LOG hLog);

/**
 * Gets the true if the LoTL is a TLS.
 * @param hLotl [in] Handle to the Lotl object.
 * @param isTsl [out] True if the LoTL is a TLS.
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLotl_GetIsTsl(ID3_BIOSEAL_LOTL hLotl, bool *isTsl);

/**
 * Returns the URL to the TSL or the next LoTL.
 * @param hLotl [in] Handle to the Lotl object.
 * @param lotlData [in] The LotL data.
 * @param lotlDataSize [in] Size of the 'lotlData' buffer.
 * @param manifestId [in] The Manifest ID in ISO 22385 format (hexadecimal string).
 * @param iac [in] The IAC (optional).
 * @param url [out] The TSL url.
 * @param urlSize [in,out] Size of the 'url' buffer. Receives the minimum required buffer size to hold the 'url' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'url' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLotl_GetUrl(ID3_BIOSEAL_LOTL hLotl, unsigned char *lotlData, int lotlDataSize, const char *manifestId, const char *iac, char *url, int *urlSize);

/**
 * Returns the URL to the manifest.
 * @param hLotl [in] Handle to the Lotl object.
 * @param tslData [in] The TSL data.
 * @param tslDataSize [in] Size of the 'tslData' buffer.
 * @param url [out] The manifest url.
 * @param urlSize [in,out] Size of the 'url' buffer. Receives the minimum required buffer size to hold the 'url' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'url' (excluding terminating NULL character).
 * @return An error code. See id3BiosealError file for the list of possible error codes.
 * @retval id3BiosealError_Success The function succeeded.
 */
ID3BIOSEAL_C_API int ID3_CDECL id3BiosealLotl_GetManifestUrl(ID3_BIOSEAL_LOTL hLotl, unsigned char *tslData, int tslDataSize, char *url, int *urlSize);

#ifdef __cplusplus
}
#endif

#endif
