//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#pragma once
#include <id3BiosealCppWrapper/id3BiosealException.hpp>
#include <id3Bioseal/id3BiosealBarcodeReader.h>
#include "vector"

namespace id3BiosealCppWrapper
{

/**
 * Provides functionality to decode barcodes from raw image data.
 */
class BarcodeReader
{
private:
    ID3_BIOSEAL_BARCODE_READER handle_{};

public:
    BarcodeReader() {
        check_error(id3BiosealBarcodeReader_Initialize(&handle_));
    }

    virtual ~BarcodeReader() {
        if (handle_) {
            id3BiosealBarcodeReader_Dispose(&handle_);
        }
    }


    BarcodeReader(BarcodeReader &&src) noexcept: handle_(src.handle_) {
        src.handle_ = nullptr;
    }

    ID3_BIOSEAL_BARCODE_READER getHandle() const { return handle_; }
    operator ID3_BIOSEAL_BARCODE_READER () const { return handle_; }

    // public getters and setters
    /**
    * Gets the holds the decoded data extracted from the barcode.
    * @param outputData [out] Holds the decoded data extracted from the barcode.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval Holds the decoded data extracted from the barcode.
    */
    void getOutputData(std::vector<unsigned char> & outputData) {
        int outputDataSize = -1;
        auto err = id3BiosealBarcodeReader_GetOutputData(handle_, outputData.data(), &outputDataSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            outputData.resize(outputDataSize);
            err = id3BiosealBarcodeReader_GetOutputData(handle_, outputData.data(), &outputDataSize);
        }
        check_error(err);
    }

    /**
    * Gets the holds the decoded data extracted from the barcode.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval Holds the decoded data extracted from the barcode.
    */
    std::vector<unsigned char> getOutputData() {
        std::vector<unsigned char> output_data;
        getOutputData(output_data);
        return output_data;
    }

    // public methods
    /**
    * Attempts to decode a barcode from the provided raw image data.
    * @param data [in] An array of bytes representing the raw image data that contains the barcode to be decoded.
    * @param dataSize [in] Size of the 'data' buffer.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval A boolean output that indicates the success of the decoding process. It is set to 'true' if the barcode is successfully decoded, otherwise 'false'.
    */
    bool decode(unsigned char *data, int dataSize) {
        bool result;
        check_error(id3BiosealBarcodeReader_Decode(handle_, data, dataSize, &result));
        return result;
    }

};
};