//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#pragma once
#include <id3BiosealCppWrapper/id3BiosealException.hpp>
#include <id3Bioseal/id3BiosealBiometricVerificationRule.h>

namespace id3BiosealCppWrapper
{

/**
 * Represents a biometric verification rule.
 */
class BiometricVerificationRule
{
private:
    ID3_BIOSEAL_BIOMETRIC_VERIFICATION_RULE handle_{};

public:
    BiometricVerificationRule() {
        check_error(id3BiosealBiometricVerificationRule_Initialize(&handle_));
    }

    virtual ~BiometricVerificationRule() {
        if (handle_) {
            id3BiosealBiometricVerificationRule_Dispose(&handle_);
        }
    }


    BiometricVerificationRule(BiometricVerificationRule &&src) noexcept: handle_(src.handle_) {
        src.handle_ = nullptr;
    }

    ID3_BIOSEAL_BIOMETRIC_VERIFICATION_RULE getHandle() const { return handle_; }
    operator ID3_BIOSEAL_BIOMETRIC_VERIFICATION_RULE () const { return handle_; }

    // public getters and setters
    /**
    * Gets the the type of biometric data to be used for verification.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The type of biometric data to be used for verification.
    */
    id3BiosealBiometricDataType getDataType() {
        id3BiosealBiometricDataType data_type;
        check_error(id3BiosealBiometricVerificationRule_GetDataType(handle_, &data_type));
        return data_type;
    }

    /**
    * Gets the the number of biometric samples of the specified type to be verified.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The number of biometric samples of the specified type to be verified.
    */
    int getCount() {
        int count;
        check_error(id3BiosealBiometricVerificationRule_GetCount(handle_, &count));
        return count;
    }

    /**
    * Gets the the biometric decision threshold.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The biometric decision threshold.
    */
    int getThreshold() {
        int threshold;
        check_error(id3BiosealBiometricVerificationRule_GetThreshold(handle_, &threshold));
        return threshold;
    }

    // public methods
};
};