//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#pragma once
#include <id3BiosealCppWrapper/id3BiosealException.hpp>
#include <id3BiosealCppWrapper/id3BiosealBiometricVerificationRule.hpp>
#include <id3Bioseal/id3BiosealBiometricVerificationRules.h>

namespace id3BiosealCppWrapper
{

/**
 * Represents a list of biometric verification rules.
 */
class BiometricVerificationRules
{
private:
    ID3_BIOSEAL_BIOMETRIC_VERIFICATION_RULES handle_{};

public:
    BiometricVerificationRules() {
        check_error(id3BiosealBiometricVerificationRules_Initialize(&handle_));
    }

    virtual ~BiometricVerificationRules() {
        if (handle_) {
            id3BiosealBiometricVerificationRules_Dispose(&handle_);
        }
    }


    BiometricVerificationRules(BiometricVerificationRules &&src) noexcept: handle_(src.handle_) {
        src.handle_ = nullptr;
    }

    ID3_BIOSEAL_BIOMETRIC_VERIFICATION_RULES getHandle() const { return handle_; }
    operator ID3_BIOSEAL_BIOMETRIC_VERIFICATION_RULES () const { return handle_; }

    // --- Iterator Support ---
    class iterator {
    private:
        BiometricVerificationRules *list_;
        int index_;
    public:
        iterator(BiometricVerificationRules *list, int index) : list_(list), index_(index) {}

        BiometricVerificationRule operator*() const
        { return list_->get(index_); }

        iterator &operator++() {
            ++index_;
            return *this;
        }

        bool operator!=(const iterator &src) const
        { return index_ != src.index_; }
    };

    iterator begin()
    { return iterator(this, 0); }

    iterator end()
    { return iterator(this, getCount()); }

    // public getters and setters
    /**
    * Gets the indicates whether all verification rules must be respected.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval Indicates whether all verification rules must be respected.
    */
    bool getVerifyAll() {
        bool verify_all;
        check_error(id3BiosealBiometricVerificationRules_GetVerifyAll(handle_, &verify_all));
        return verify_all;
    }

    // public methods
    /**
    * Gets an item of the BiometricVerificationRules object.
    * @param index [in] Index of the BiometricVerificationRule item to get.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval BiometricVerificationRule item to get.
    */
    BiometricVerificationRule get(int index) {
        BiometricVerificationRule biometric_verification_rule_item;
        check_error(id3BiosealBiometricVerificationRules_Get(handle_, index, biometric_verification_rule_item));
        return biometric_verification_rule_item;
    }

    /**
    * Gets the number of elements in the BiometricVerificationRules object.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval Number of elements in the BiometricVerificationRule object.
    */
    int getCount() {
        int count;
        check_error(id3BiosealBiometricVerificationRules_GetCount(handle_, &count));
        return count;
    }

};
};