//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#pragma once
#include <id3BiosealCppWrapper/id3BiosealException.hpp>
#include <id3Bioseal/id3BiosealCertificateExtension.h>
#include "vector"

namespace id3BiosealCppWrapper
{

/**
 * Information about a certificate extension.
 */
class CertificateExtension
{
private:
    ID3_BIOSEAL_CERTIFICATE_EXTENSION handle_{};

public:
    CertificateExtension() {
        check_error(id3BiosealCertificateExtension_Initialize(&handle_));
    }

    virtual ~CertificateExtension() {
        if (handle_) {
            id3BiosealCertificateExtension_Dispose(&handle_);
        }
    }


    CertificateExtension(CertificateExtension &&src) noexcept: handle_(src.handle_) {
        src.handle_ = nullptr;
    }

    ID3_BIOSEAL_CERTIFICATE_EXTENSION getHandle() const { return handle_; }
    operator ID3_BIOSEAL_CERTIFICATE_EXTENSION () const { return handle_; }

    // public getters and setters
    /**
    * Gets the the OID of the certificate extension.
    * @param oid [out] The OID of the certificate extension.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The OID of the certificate extension.
    */
    void getOid(std::string & oid) {
        int oidSize = -1;
        auto err = id3BiosealCertificateExtension_GetOid(handle_, oid.data(), &oidSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            oid.resize(oidSize);
            err = id3BiosealCertificateExtension_GetOid(handle_, oid.data(), &oidSize);
            if (err == 0) { oid.resize(oidSize); }
        }
        check_error(err);
    }

    /**
    * Gets the the OID of the certificate extension.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The OID of the certificate extension.
    */
    std::string getOid() {
        std::string oid;
        getOid(oid);
        return oid;
    }

    /**
    * Gets the A byte array that contains the value of the certificate extension.
    * @param value [out] A byte array that contains the value of the certificate extension.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval A byte array that contains the value of the certificate extension.
    */
    void getValue(std::vector<unsigned char> & value) {
        int valueSize = -1;
        auto err = id3BiosealCertificateExtension_GetValue(handle_, value.data(), &valueSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            value.resize(valueSize);
            err = id3BiosealCertificateExtension_GetValue(handle_, value.data(), &valueSize);
        }
        check_error(err);
    }

    /**
    * Gets the A byte array that contains the value of the certificate extension.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval A byte array that contains the value of the certificate extension.
    */
    std::vector<unsigned char> getValue() {
        std::vector<unsigned char> value;
        getValue(value);
        return value;
    }

    /**
    * Gets the A value indicating whether the certificate extension is critical.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval A value indicating whether the certificate extension is critical.
    */
    bool getCritical() {
        bool critical;
        check_error(id3BiosealCertificateExtension_GetCritical(handle_, &critical));
        return critical;
    }

    // public methods
};
};