//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#pragma once
#include <id3BiosealCppWrapper/id3BiosealException.hpp>
#include <id3BiosealCppWrapper/id3BiosealCertificateExtension.hpp>
#include <id3Bioseal/id3BiosealCertificateExtensionList.h>

namespace id3BiosealCppWrapper
{

/**
 * Represents a list of certificate extensions.
 */
class CertificateExtensionList
{
private:
    ID3_BIOSEAL_CERTIFICATE_EXTENSION_LIST handle_{};

public:
    CertificateExtensionList() {
        check_error(id3BiosealCertificateExtensionList_Initialize(&handle_));
    }

    virtual ~CertificateExtensionList() {
        if (handle_) {
            id3BiosealCertificateExtensionList_Dispose(&handle_);
        }
    }


    CertificateExtensionList(CertificateExtensionList &&src) noexcept: handle_(src.handle_) {
        src.handle_ = nullptr;
    }

    ID3_BIOSEAL_CERTIFICATE_EXTENSION_LIST getHandle() const { return handle_; }
    operator ID3_BIOSEAL_CERTIFICATE_EXTENSION_LIST () const { return handle_; }

    // --- Iterator Support ---
    class iterator {
    private:
        CertificateExtensionList *list_;
        int index_;
    public:
        iterator(CertificateExtensionList *list, int index) : list_(list), index_(index) {}

        CertificateExtension operator*() const
        { return list_->get(index_); }

        iterator &operator++() {
            ++index_;
            return *this;
        }

        bool operator!=(const iterator &src) const
        { return index_ != src.index_; }
    };

    iterator begin()
    { return iterator(this, 0); }

    iterator end()
    { return iterator(this, getCount()); }

    // public getters and setters
    // public methods
    /**
    * Gets an item of the CertificateExtensionList object.
    * @param index [in] Index of the CertificateExtension item to get.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval CertificateExtension item to get.
    */
    CertificateExtension get(int index) {
        CertificateExtension certificate_extension_item;
        check_error(id3BiosealCertificateExtensionList_Get(handle_, index, certificate_extension_item));
        return certificate_extension_item;
    }

    /**
    * Gets the number of elements in the CertificateExtensionList object.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval Number of elements in the CertificateExtension object.
    */
    int getCount() {
        int count;
        check_error(id3BiosealCertificateExtensionList_GetCount(handle_, &count));
        return count;
    }

};
};