//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#pragma once
#include <id3BiosealCppWrapper/id3BiosealException.hpp>
#include <id3BiosealCppWrapper/id3BiosealDateTime.hpp>
#include <id3Bioseal/id3BiosealDateTimeArray.h>

namespace id3BiosealCppWrapper
{

/**
 * Represents an array of DateTime objects.
 */
class DateTimeArray
{
private:
    ID3_BIOSEAL_DATE_TIME_ARRAY handle_{};

public:
    DateTimeArray() {
        check_error(id3BiosealDateTimeArray_Initialize(&handle_));
    }

    virtual ~DateTimeArray() {
        if (handle_) {
            id3BiosealDateTimeArray_Dispose(&handle_);
        }
    }


    DateTimeArray(DateTimeArray &&src) noexcept: handle_(src.handle_) {
        src.handle_ = nullptr;
    }

    ID3_BIOSEAL_DATE_TIME_ARRAY getHandle() const { return handle_; }
    operator ID3_BIOSEAL_DATE_TIME_ARRAY () const { return handle_; }

    // --- Iterator Support ---
    class iterator {
    private:
        DateTimeArray *list_;
        int index_;
    public:
        iterator(DateTimeArray *list, int index) : list_(list), index_(index) {}

        DateTime operator*() const
        { return list_->get(index_); }

        iterator &operator++() {
            ++index_;
            return *this;
        }

        bool operator!=(const iterator &src) const
        { return index_ != src.index_; }
    };

    iterator begin()
    { return iterator(this, 0); }

    iterator end()
    { return iterator(this, getCount()); }

    // public getters and setters
    // public methods
    /**
    * Gets an item of the DateTimeArray object.
    * @param index [in] Index of the DateTime item to get.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval DateTime item to get.
    */
    DateTime get(int index) {
        DateTime date_time_item;
        check_error(id3BiosealDateTimeArray_Get(handle_, index, date_time_item));
        return date_time_item;
    }

    /**
    * Gets the number of elements in the DateTimeArray object.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval Number of elements in the DateTime object.
    */
    int getCount() {
        int count;
        check_error(id3BiosealDateTimeArray_GetCount(handle_, &count));
        return count;
    }

};
};