//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#pragma once
#include <id3BiosealCppWrapper/id3BiosealException.hpp>
#include <id3BiosealCppWrapper/id3BiosealStringArray.hpp>
#include <id3BiosealCppWrapper/id3BiosealExtension.hpp>
#include <id3Bioseal/id3BiosealExtensionList.h>

namespace id3BiosealCppWrapper
{

/**
 * Represents a list of extensions.
 */
class ExtensionList
{
private:
    ID3_BIOSEAL_EXTENSION_LIST handle_{};

public:
    ExtensionList() {
        check_error(id3BiosealExtensionList_Initialize(&handle_));
    }

    virtual ~ExtensionList() {
        if (handle_) {
            id3BiosealExtensionList_Dispose(&handle_);
        }
    }


    ExtensionList(ExtensionList &&src) noexcept: handle_(src.handle_) {
        src.handle_ = nullptr;
    }

    ID3_BIOSEAL_EXTENSION_LIST getHandle() const { return handle_; }
    operator ID3_BIOSEAL_EXTENSION_LIST () const { return handle_; }

    // public getters and setters
    /**
    * Gets the gets a string list containing the keys in the dict.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval Gets a string list containing the keys in the dict.
    */
    StringArray getKeys() {
        StringArray keys;
        check_error(id3BiosealExtensionList_GetKeys(handle_, keys));
        return keys;
    }

    // public methods
    /**
    * Gets an item of the ExtensionList object.
    * @param key [in] Unique key of the Extension item to get.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval Extension item to get.
    */
    Extension get(const char *key) {
        Extension extension_item;
        check_error(id3BiosealExtensionList_Get(handle_, key, extension_item));
        return extension_item;
    }

    /**
    * Gets the number of elements in the ExtensionList object.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval Number of elements in the Extension object.
    */
    int getCount() {
        int count;
        check_error(id3BiosealExtensionList_GetCount(handle_, &count));
        return count;
    }

    /**
    * Determines whether the ExtensionListobject contains the specified key.
    * @param key [in] The key to locate in the Extension object.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval true if the Extension object contains an element with the specified key, otherwise false.
    */
    bool containsKey(const char *key) {
        bool result;
        check_error(id3BiosealExtensionList_ContainsKey(handle_, key, &result));
        return result;
    }

    /**
    * Clear the list.
    * @exception BiosealException An error has occured during Bioseal library execution.
    */
    void clear() {
        check_error(id3BiosealExtensionList_Clear(handle_));
    }

};
};