//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#pragma once
#include <id3BiosealCppWrapper/id3BiosealException.hpp>
#include <id3BiosealCppWrapper/id3BiosealField.hpp>
#include <id3Bioseal/id3BiosealFieldList.h>

namespace id3BiosealCppWrapper
{

/**
 * Represents a list of Field objects.
 */
class FieldList
{
private:
    ID3_BIOSEAL_FIELD_LIST handle_{};

public:
    FieldList() {
        check_error(id3BiosealFieldList_Initialize(&handle_));
    }

    virtual ~FieldList() {
        if (handle_) {
            id3BiosealFieldList_Dispose(&handle_);
        }
    }


    FieldList(FieldList &&src) noexcept: handle_(src.handle_) {
        src.handle_ = nullptr;
    }

    ID3_BIOSEAL_FIELD_LIST getHandle() const { return handle_; }
    operator ID3_BIOSEAL_FIELD_LIST () const { return handle_; }

    // --- Iterator Support ---
    class iterator {
    private:
        FieldList *list_;
        int index_;
    public:
        iterator(FieldList *list, int index) : list_(list), index_(index) {}

        Field operator*() const
        { return list_->get(index_); }

        iterator &operator++() {
            ++index_;
            return *this;
        }

        bool operator!=(const iterator &src) const
        { return index_ != src.index_; }
    };

    iterator begin()
    { return iterator(this, 0); }

    iterator end()
    { return iterator(this, getCount()); }

    // public getters and setters
    // public methods
    /**
    * Gets an item of the FieldList object.
    * @param index [in] Index of the Field item to get.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval Field item to get.
    */
    Field get(int index) {
        Field field_item;
        check_error(id3BiosealFieldList_Get(handle_, index, field_item));
        return field_item;
    }

    /**
    * Gets the number of elements in the FieldList object.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval Number of elements in the Field object.
    */
    int getCount() {
        int count;
        check_error(id3BiosealFieldList_GetCount(handle_, &count));
        return count;
    }

    /**
    * Clear the list.
    * @exception BiosealException An error has occured during Bioseal library execution.
    */
    void clear() {
        check_error(id3BiosealFieldList_Clear(handle_));
    }

};
};