//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#pragma once
#include <id3BiosealCppWrapper/id3BiosealException.hpp>
#include <id3Bioseal/id3BiosealLicense.h>
#include "vector"

namespace id3BiosealCppWrapper
{

/**
 * Handles the license to use this SDK.
 */
class BiosealLicense
{
private:
    // Disallow creating an instance of this object
    BiosealLicense() {}

public:
    // public getters and setters
    // public methods
    /**
    * Checks a license from a file.
    * This function does two things:
    * - First it loads the license from a file using the given path
    * - Then it checks the validity of the license regarding the host it is called on (Windows, Linux, Android ...)
    * Important: Calling this function is required to get access to the license member getters.
    * Note: The optional parameter is required on Android and must be the JNIEnv* pointer casted as a void*. It can be set as NULL for other devices.
    * @param licensePath [in] Absolute or relative path to the license file.
    * @param optionalParameter [in] Only used on Android: JNIEnv* pointer casted as a void*. Ignored for other devices.
    * @exception BiosealException An error has occured during Bioseal library execution.
    */
    static void checkLicense(const char *licensePath, void *optionalParameter=nullptr) {
        check_error(id3BiosealLicense_CheckLicense(licensePath, optionalParameter));
    }

    /**
    * Checks a license from a buffer.
    * This function does two things:
    * - First it loads the license from a buffer
    * - Then it checks the validity of the license regarding the host it is called on (Windows, Linux, Android ...)
    * Important: Calling one of the license checking function is required to get access to all the license member getters.
    * Note: The optional parameter is required on Android and must be the JNIEnv* pointer casted as a void*. It can be set as NULL for other devices.
    * @param licenseData [in] A buffer containing the license file data.
    * @param licenseDataSize [in] Size of the 'licenseData' buffer.
    * @param optionalParameter [in] Only used on Android: JNIEnv* pointer casted as a void*. Ignored for other devices.
    * @exception BiosealException An error has occured during Bioseal library execution.
    */
    static void checkLicenseBuffer(unsigned char *licenseData, int licenseDataSize, void *optionalParameter=nullptr) {
        check_error(id3BiosealLicense_CheckLicenseBuffer(licenseData, licenseDataSize, optionalParameter));
    }

    /**
    * Checks if a module is present in the previously loaded license.
    * @param moduleName [in] The module name.
    * @exception BiosealException An error has occured during Bioseal library execution.
    */
    static void checkModule(const char *moduleName) {
        check_error(id3BiosealLicense_CheckModule(moduleName));
    }

    /**
    * Retrieves the number of modules in the license.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The number of modules in the license.
    */
    static int getModuleCount() {
        int count;
        check_error(id3BiosealLicense_GetModuleCount(&count));
        return count;
    }

    /**
    * Retrieves the name of the specified license module.
    * @param index [in] Index of the module, starting from 0.
    * @param name [out] The license module name.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license module name.
    */
    static void getModuleName(int index, std::string & name) {
        int nameSize = -1;
        auto err = id3BiosealLicense_GetModuleName(index, name.data(), &nameSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            name.resize(nameSize);
            err = id3BiosealLicense_GetModuleName(index, name.data(), &nameSize);
            if (err == 0) { name.resize(nameSize); }
        }
        check_error(err);
    }

    /**
    * Retrieves the name of the specified license module.
    * @param index [in] Index of the module, starting from 0.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license module name.
    */
    static std::string getModuleName(int index) {
        std::string name;
        getModuleName(index, name);
        return name;
    }

    /**
    * Retrieves the value associated to the specified license module.
    * @param name [in] Name of the requested module.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The value associated to the licence module.
    */
    static int getModuleValue(const char *name) {
        int value;
        check_error(id3BiosealLicense_GetModuleValue(name, &value));
        return value;
    }

    /**
    * Retrieves the license name.
    * @param licenseName [out] The license name.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license name.
    */
    static void getLicenseName(std::string & licenseName) {
        int licenseNameSize = -1;
        auto err = id3BiosealLicense_GetLicenseName(licenseName.data(), &licenseNameSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            licenseName.resize(licenseNameSize);
            err = id3BiosealLicense_GetLicenseName(licenseName.data(), &licenseNameSize);
            if (err == 0) { licenseName.resize(licenseNameSize); }
        }
        check_error(err);
    }

    /**
    * Retrieves the license name.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license name.
    */
    static std::string getLicenseName() {
        std::string license_name;
        getLicenseName(license_name);
        return license_name;
    }

    /**
    * Retrieves the license type.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license type.
    */
    static id3BiosealLicenseType getLicenseType() {
        id3BiosealLicenseType bioseal_license_type;
        check_error(id3BiosealLicense_GetLicenseType(&bioseal_license_type));
        return bioseal_license_type;
    }

    /**
    * Retrieves the license owner.
    * @param licenseOwner [out] The license owner.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license owner.
    */
    static void getLicenseOwner(std::string & licenseOwner) {
        int licenseOwnerSize = -1;
        auto err = id3BiosealLicense_GetLicenseOwner(licenseOwner.data(), &licenseOwnerSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            licenseOwner.resize(licenseOwnerSize);
            err = id3BiosealLicense_GetLicenseOwner(licenseOwner.data(), &licenseOwnerSize);
            if (err == 0) { licenseOwner.resize(licenseOwnerSize); }
        }
        check_error(err);
    }

    /**
    * Retrieves the license owner.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license owner.
    */
    static std::string getLicenseOwner() {
        std::string license_owner;
        getLicenseOwner(license_owner);
        return license_owner;
    }

    /**
    * Retrieves the license product reference.
    * @param productReference [out] The product reference.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The product reference.
    */
    static void getProductReference(std::string & productReference) {
        int productReferenceSize = -1;
        auto err = id3BiosealLicense_GetProductReference(productReference.data(), &productReferenceSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            productReference.resize(productReferenceSize);
            err = id3BiosealLicense_GetProductReference(productReference.data(), &productReferenceSize);
            if (err == 0) { productReference.resize(productReferenceSize); }
        }
        check_error(err);
    }

    /**
    * Retrieves the license product reference.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The product reference.
    */
    static std::string getProductReference() {
        std::string product_reference;
        getProductReference(product_reference);
        return product_reference;
    }

    /**
    * Retrieves the license path.
    * @param licensePath [out] The license path.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license path.
    */
    static void getLicensePath(std::string & licensePath) {
        int licensePathSize = -1;
        auto err = id3BiosealLicense_GetLicensePath(licensePath.data(), &licensePathSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            licensePath.resize(licensePathSize);
            err = id3BiosealLicense_GetLicensePath(licensePath.data(), &licensePathSize);
            if (err == 0) { licensePath.resize(licensePathSize); }
        }
        check_error(err);
    }

    /**
    * Retrieves the license path.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license path.
    */
    static std::string getLicensePath() {
        std::string license_path;
        getLicensePath(license_path);
        return license_path;
    }

    /**
    * Retrieves the number of remaining days.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The number of remaining days.
    */
    static int getRemainingDays() {
        int remaining_days;
        check_error(id3BiosealLicense_GetRemainingDays(&remaining_days));
        return remaining_days;
    }

    /**
    * Retrieves the license expiry date.
    * @param expiryDate [out] The license expiry date.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license expiry date.
    */
    static void getExpiryDate(std::string & expiryDate) {
        int expiryDateSize = -1;
        auto err = id3BiosealLicense_GetExpiryDate(expiryDate.data(), &expiryDateSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            expiryDate.resize(expiryDateSize);
            err = id3BiosealLicense_GetExpiryDate(expiryDate.data(), &expiryDateSize);
            if (err == 0) { expiryDate.resize(expiryDateSize); }
        }
        check_error(err);
    }

    /**
    * Retrieves the license expiry date.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license expiry date.
    */
    static std::string getExpiryDate() {
        std::string expiry_date;
        getExpiryDate(expiry_date);
        return expiry_date;
    }

    /**
    * Retrieves the hardware code of the previously loaded license.
    * @param licenseFileHardwareCode [out] The license hardware code.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license hardware code.
    */
    static void getLicenseFileHardwareCode(std::string & licenseFileHardwareCode) {
        int licenseFileHardwareCodeSize = -1;
        auto err = id3BiosealLicense_GetLicenseFileHardwareCode(licenseFileHardwareCode.data(), &licenseFileHardwareCodeSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            licenseFileHardwareCode.resize(licenseFileHardwareCodeSize);
            err = id3BiosealLicense_GetLicenseFileHardwareCode(licenseFileHardwareCode.data(), &licenseFileHardwareCodeSize);
            if (err == 0) { licenseFileHardwareCode.resize(licenseFileHardwareCodeSize); }
        }
        check_error(err);
    }

    /**
    * Retrieves the hardware code of the previously loaded license.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license hardware code.
    */
    static std::string getLicenseFileHardwareCode() {
        std::string license_file_hardware_code;
        getLicenseFileHardwareCode(license_file_hardware_code);
        return license_file_hardware_code;
    }

    /**
    * Retrieves the serial number of the previously loaded license.
    * @param licenseFileSerial [out] The license serial number.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license serial number.
    */
    static void getLicenseFileSerial(std::string & licenseFileSerial) {
        int licenseFileSerialSize = -1;
        auto err = id3BiosealLicense_GetLicenseFileSerial(licenseFileSerial.data(), &licenseFileSerialSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            licenseFileSerial.resize(licenseFileSerialSize);
            err = id3BiosealLicense_GetLicenseFileSerial(licenseFileSerial.data(), &licenseFileSerialSize);
            if (err == 0) { licenseFileSerial.resize(licenseFileSerialSize); }
        }
        check_error(err);
    }

    /**
    * Retrieves the serial number of the previously loaded license.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license serial number.
    */
    static std::string getLicenseFileSerial() {
        std::string license_file_serial;
        getLicenseFileSerial(license_file_serial);
        return license_file_serial;
    }

    /**
    * Retrieves the hardware code of the device using the requested hardware code type.
    * This function retrieves the hardware code of the device you run the library on.
    * The optional parameter is required:
    * - On Android, it must be the JNIEnv* pointer cast as a void*.
    * - In other cases, it must be set to NULL.
    * @param hardwareCodeType [in] The requested hardware code type.
    * @param code [out] The device hardware code.
    * @param optionalParameter [in] Only used on Android for JNIEnv* pointer. Ignored for other devices (NULL).
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The device hardware code.
    */
    static void getHostHardwareCode(id3BiosealLicenseHardwareCodeType hardwareCodeType, std::string & code, void *optionalParameter=nullptr) {
        int codeSize = -1;
        auto err = id3BiosealLicense_GetHostHardwareCode(hardwareCodeType, code.data(), &codeSize, optionalParameter);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            code.resize(codeSize);
            err = id3BiosealLicense_GetHostHardwareCode(hardwareCodeType, code.data(), &codeSize, optionalParameter);
            if (err == 0) { code.resize(codeSize); }
        }
        check_error(err);
    }

    /**
    * Retrieves the hardware code of the device using the requested hardware code type.
    * This function retrieves the hardware code of the device you run the library on.
    * The optional parameter is required:
    * - On Android, it must be the JNIEnv* pointer cast as a void*.
    * - In other cases, it must be set to NULL.
    * @param hardwareCodeType [in] The requested hardware code type.
    * @param optionalParameter [in] Only used on Android for JNIEnv* pointer. Ignored for other devices (NULL).
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The device hardware code.
    */
    static std::string getHostHardwareCode(id3BiosealLicenseHardwareCodeType hardwareCodeType, void *optionalParameter=nullptr) {
        std::string code;
        getHostHardwareCode(hardwareCodeType, code, optionalParameter);
        return code;
    }

    /**
    * Retrieves the hardware code of the external drive you run the library on. This function works on Windows only.
    * @param hardwareCodeType [in] The requested hardware code type. Must be windowsUsb.
    * @param code [out] The device hardware code.
    * @param externalDrivePath [in] On Windows to activate a license on a USB device, it must contain the path to the requested USB drive (e.g 'G:\').
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The device hardware code.
    */
    static void getExternalDriveHardwareCode(id3BiosealLicenseHardwareCodeType hardwareCodeType, std::string & code, const char *externalDrivePath=nullptr) {
        int codeSize = -1;
        auto err = id3BiosealLicense_GetExternalDriveHardwareCode(hardwareCodeType, code.data(), &codeSize, externalDrivePath);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            code.resize(codeSize);
            err = id3BiosealLicense_GetExternalDriveHardwareCode(hardwareCodeType, code.data(), &codeSize, externalDrivePath);
            if (err == 0) { code.resize(codeSize); }
        }
        check_error(err);
    }

    /**
    * Retrieves the hardware code of the external drive you run the library on. This function works on Windows only.
    * @param hardwareCodeType [in] The requested hardware code type. Must be windowsUsb.
    * @param externalDrivePath [in] On Windows to activate a license on a USB device, it must contain the path to the requested USB drive (e.g 'G:\').
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The device hardware code.
    */
    static std::string getExternalDriveHardwareCode(id3BiosealLicenseHardwareCodeType hardwareCodeType, const char *externalDrivePath=nullptr) {
        std::string code;
        getExternalDriveHardwareCode(hardwareCodeType, code, externalDrivePath);
        return code;
    }

    /**
    * Retrieves the license comment.
    * @param licenseComment [out] The license comment.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license comment.
    */
    static void getLicenseComment(std::string & licenseComment) {
        int licenseCommentSize = -1;
        auto err = id3BiosealLicense_GetLicenseComment(licenseComment.data(), &licenseCommentSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            licenseComment.resize(licenseCommentSize);
            err = id3BiosealLicense_GetLicenseComment(licenseComment.data(), &licenseCommentSize);
            if (err == 0) { licenseComment.resize(licenseCommentSize); }
        }
        check_error(err);
    }

    /**
    * Retrieves the license comment.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license comment.
    */
    static std::string getLicenseComment() {
        std::string license_comment;
        getLicenseComment(license_comment);
        return license_comment;
    }

    /**
    * Activates a license using customer credentials and a product reference and saves the license to a file.
    * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
    * @param login [in] The id3 account login.
    * @param password [in] The id3 account password.
    * @param productReference [in] The requested product reference.
    * @param commentary [in] Commentary associated to this license activation.
    * @param path [in] Path to the retrieved license file.
    * @exception BiosealException An error has occured during Bioseal library execution.
    */
    static void activate(const char *hardwareCode, const char *login, const char *password, const char *productReference, const char *commentary, const char *path) {
        check_error(id3BiosealLicense_Activate(hardwareCode, login, password, productReference, commentary, path));
    }

    /**
    * Retrieves a license file buffer using customer credentials and a product reference and returns the license in a data buffer.
    * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
    * @param login [in] The customer login.
    * @param password [in] The customer password.
    * @param productReference [in] The requested product reference.
    * @param commentary [in] Commentary associated to this license activation, generally the host name.
    * @param licenseFileBuffer [out] The license data.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license data.
    */
    static void activateBuffer(const char *hardwareCode, const char *login, const char *password, const char *productReference, const char *commentary, std::vector<unsigned char> & licenseFileBuffer) {
        int licenseFileBufferSize = -1;
        auto err = id3BiosealLicense_ActivateBuffer(hardwareCode, login, password, productReference, commentary, licenseFileBuffer.data(), &licenseFileBufferSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            licenseFileBuffer.resize(licenseFileBufferSize);
            err = id3BiosealLicense_ActivateBuffer(hardwareCode, login, password, productReference, commentary, licenseFileBuffer.data(), &licenseFileBufferSize);
        }
        check_error(err);
    }

    /**
    * Retrieves a license file buffer using customer credentials and a product reference and returns the license in a data buffer.
    * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
    * @param login [in] The customer login.
    * @param password [in] The customer password.
    * @param productReference [in] The requested product reference.
    * @param commentary [in] Commentary associated to this license activation, generally the host name.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license data.
    */
    static std::vector<unsigned char> activateBuffer(const char *hardwareCode, const char *login, const char *password, const char *productReference, const char *commentary) {
        std::vector<unsigned char> license_file_buffer;
        activateBuffer(hardwareCode, login, password, productReference, commentary, license_file_buffer);
        return license_file_buffer;
    }

    /**
    * Activates a license using a serial key and saves the license to a file.
    * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
    * @param serialKey [in] The serial key to activate.
    * @param commentary [in] Commentary associated to this license activation, generally the host name.
    * @param path [in] Path to the license file.
    * @exception BiosealException An error has occured during Bioseal library execution.
    */
    static void activateSerialKey(const char *hardwareCode, const char *serialKey, const char *commentary, const char *path) {
        check_error(id3BiosealLicense_ActivateSerialKey(hardwareCode, serialKey, commentary, path));
    }

    /**
    * Activates a license using a serial key and returns the license in a data buffer.
    * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
    * @param serialKey [in] The id3 serial key.
    * @param commentary [in] Commentary associated to this license activation.
    * @param licenseFileBuffer [out] The license data buffer.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license data buffer.
    */
    static void activateSerialKeyBuffer(const char *hardwareCode, const char *serialKey, const char *commentary, std::vector<unsigned char> & licenseFileBuffer) {
        int licenseFileBufferSize = -1;
        auto err = id3BiosealLicense_ActivateSerialKeyBuffer(hardwareCode, serialKey, commentary, licenseFileBuffer.data(), &licenseFileBufferSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            licenseFileBuffer.resize(licenseFileBufferSize);
            err = id3BiosealLicense_ActivateSerialKeyBuffer(hardwareCode, serialKey, commentary, licenseFileBuffer.data(), &licenseFileBufferSize);
        }
        check_error(err);
    }

    /**
    * Activates a license using a serial key and returns the license in a data buffer.
    * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
    * @param serialKey [in] The id3 serial key.
    * @param commentary [in] Commentary associated to this license activation.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license data buffer.
    */
    static std::vector<unsigned char> activateSerialKeyBuffer(const char *hardwareCode, const char *serialKey, const char *commentary) {
        std::vector<unsigned char> license_file_buffer;
        activateSerialKeyBuffer(hardwareCode, serialKey, commentary, license_file_buffer);
        return license_file_buffer;
    }

    /**
    * Activates a license file using an activation key and saves the license to a file.
    * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
    * @param activationKey [in] The activation key.
    * @param commentary [in] Commentary associated to this license activation, generally the host name.
    * @param path [in] Path to the retrieved license file.
    * @exception BiosealException An error has occured during Bioseal library execution.
    */
    static void activateActivationKey(const char *hardwareCode, const char *activationKey, const char *commentary, const char *path) {
        check_error(id3BiosealLicense_ActivateActivationKey(hardwareCode, activationKey, commentary, path));
    }

    /**
    * Activates a license using an activation key and returns the license in a data buffer.
    * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
    * @param activationKey [in] The id3 activation key.
    * @param commentary [in] Commentary associated to this license activation.
    * @param licenseFileBuffer [out] The license data buffer.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license data buffer.
    */
    static void activateActivationKeyBuffer(const char *hardwareCode, const char *activationKey, const char *commentary, std::vector<unsigned char> & licenseFileBuffer) {
        int licenseFileBufferSize = -1;
        auto err = id3BiosealLicense_ActivateActivationKeyBuffer(hardwareCode, activationKey, commentary, licenseFileBuffer.data(), &licenseFileBufferSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            licenseFileBuffer.resize(licenseFileBufferSize);
            err = id3BiosealLicense_ActivateActivationKeyBuffer(hardwareCode, activationKey, commentary, licenseFileBuffer.data(), &licenseFileBufferSize);
        }
        check_error(err);
    }

    /**
    * Activates a license using an activation key and returns the license in a data buffer.
    * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
    * @param activationKey [in] The id3 activation key.
    * @param commentary [in] Commentary associated to this license activation.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The license data buffer.
    */
    static std::vector<unsigned char> activateActivationKeyBuffer(const char *hardwareCode, const char *activationKey, const char *commentary) {
        std::vector<unsigned char> license_file_buffer;
        activateActivationKeyBuffer(hardwareCode, activationKey, commentary, license_file_buffer);
        return license_file_buffer;
    }

    /**
    * Reactivates a license file using the host hardware code and the product reference.
    * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
    * @param productReference [in] The product reference.
    * @param path [in] Path to the license file.
    * @exception BiosealException An error has occured during Bioseal library execution.
    */
    static void reactivate(const char *hardwareCode, const char *productReference, const char *path) {
        check_error(id3BiosealLicense_Reactivate(hardwareCode, productReference, path));
    }

    /**
    * Reactivates a license in a data buffer using the host hardware code and the product reference.
    * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
    * @param productReference [in] The product reference.
    * @param licenseFileBuffer [out] The new license data buffer.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The new license data buffer.
    */
    static void reactivateBuffer(const char *hardwareCode, const char *productReference, std::vector<unsigned char> & licenseFileBuffer) {
        int licenseFileBufferSize = -1;
        auto err = id3BiosealLicense_ReactivateBuffer(hardwareCode, productReference, licenseFileBuffer.data(), &licenseFileBufferSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            licenseFileBuffer.resize(licenseFileBufferSize);
            err = id3BiosealLicense_ReactivateBuffer(hardwareCode, productReference, licenseFileBuffer.data(), &licenseFileBufferSize);
        }
        check_error(err);
    }

    /**
    * Reactivates a license in a data buffer using the host hardware code and the product reference.
    * @param hardwareCode [in] The hardware code on which the license file will be attached, to be retrieved using the GetHostHardwareCode function.
    * @param productReference [in] The product reference.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The new license data buffer.
    */
    static std::vector<unsigned char> reactivateBuffer(const char *hardwareCode, const char *productReference) {
        std::vector<unsigned char> license_file_buffer;
        reactivateBuffer(hardwareCode, productReference, license_file_buffer);
        return license_file_buffer;
    }

    /**
    * Checks the license by using a card-based verification process.
    * It takes the path to the license file and outputs a challenge response for further verification.
    * @param licensePath [in] The file path to the license that needs to be checked.
    * @param challengeR1 [out] An array to hold the challenge response generated by the host.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval An array to hold the challenge response generated by the host.
    */
    static void checkLicenseWithCard(const char *licensePath, std::vector<unsigned char> & challengeR1) {
        int challengeR1Size = -1;
        auto err = id3BiosealLicense_CheckLicenseWithCard(licensePath, challengeR1.data(), &challengeR1Size);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            challengeR1.resize(challengeR1Size);
            err = id3BiosealLicense_CheckLicenseWithCard(licensePath, challengeR1.data(), &challengeR1Size);
        }
        check_error(err);
    }

    /**
    * Checks the license by using a card-based verification process.
    * It takes the path to the license file and outputs a challenge response for further verification.
    * @param licensePath [in] The file path to the license that needs to be checked.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval An array to hold the challenge response generated by the host.
    */
    static std::vector<unsigned char> checkLicenseWithCard(const char *licensePath) {
        std::vector<unsigned char> challenge_r1;
        checkLicenseWithCard(licensePath, challenge_r1);
        return challenge_r1;
    }

    /**
    * Checks the license using a buffer containing the license data and a card-based verification process.
    * It outputs a challenge response for further verification.
    * @param licenseData [in] A buffer containing the license file data.
    * @param licenseDataSize [in] Size of the 'licenseData' buffer.
    * @param challengeR1 [out] An array to hold the challenge response generated by the host.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval An array to hold the challenge response generated by the host.
    */
    static void checkLicenseBufferWithCard(unsigned char *licenseData, int licenseDataSize, std::vector<unsigned char> & challengeR1) {
        int challengeR1Size = -1;
        auto err = id3BiosealLicense_CheckLicenseBufferWithCard(licenseData, licenseDataSize, challengeR1.data(), &challengeR1Size);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            challengeR1.resize(challengeR1Size);
            err = id3BiosealLicense_CheckLicenseBufferWithCard(licenseData, licenseDataSize, challengeR1.data(), &challengeR1Size);
        }
        check_error(err);
    }

    /**
    * Checks the license using a buffer containing the license data and a card-based verification process.
    * It outputs a challenge response for further verification.
    * @param licenseData [in] A buffer containing the license file data.
    * @param licenseDataSize [in] Size of the 'licenseData' buffer.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval An array to hold the challenge response generated by the host.
    */
    static std::vector<unsigned char> checkLicenseBufferWithCard(unsigned char *licenseData, int licenseDataSize) {
        std::vector<unsigned char> challenge_r1;
        checkLicenseBufferWithCard(licenseData, licenseDataSize, challenge_r1);
        return challenge_r1;
    }

    /**
    * Finalizes the license check process using the provided cryptogram.
    * It ensures the integrity and authenticity of the license verification.
    * @param cryptogram [in] A 32-byte array containing the cryptogram used for final verification.
    * @param cryptogramSize [in] Size of the 'cryptogram' buffer.
    * @exception BiosealException An error has occured during Bioseal library execution.
    */
    static void finalizeCheckLicenseWithCard(unsigned char *cryptogram, int cryptogramSize) {
        check_error(id3BiosealLicense_FinalizeCheckLicenseWithCard(cryptogram, cryptogramSize));
    }

    /**
    * Encrypt a license from XML source.
    * @param magicCode [in] Access key.
    * @param src [in] XML source.
    * @param licenseData [out] Encrypted license data.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval Encrypted license data.
    */
    static void encrypt(const char *magicCode, const char *src, std::vector<unsigned char> & licenseData) {
        int licenseDataSize = -1;
        auto err = id3BiosealLicense_Encrypt(magicCode, src, licenseData.data(), &licenseDataSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            licenseData.resize(licenseDataSize);
            err = id3BiosealLicense_Encrypt(magicCode, src, licenseData.data(), &licenseDataSize);
        }
        check_error(err);
    }

    /**
    * Encrypt a license from XML source.
    * @param magicCode [in] Access key.
    * @param src [in] XML source.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval Encrypted license data.
    */
    static std::vector<unsigned char> encrypt(const char *magicCode, const char *src) {
        std::vector<unsigned char> license_data;
        encrypt(magicCode, src, license_data);
        return license_data;
    }

    /**
    * Build an application hardware code.
    * @param applicationName [in] The application name.
    * @param certificate [in] The certificate.
    * @param code [out] The application hardware code.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The application hardware code.
    */
    static void getApplicationHardwareCode(const char *applicationName, const char *certificate, std::string & code) {
        int codeSize = -1;
        auto err = id3BiosealLicense_GetApplicationHardwareCode(applicationName, certificate, code.data(), &codeSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            code.resize(codeSize);
            err = id3BiosealLicense_GetApplicationHardwareCode(applicationName, certificate, code.data(), &codeSize);
            if (err == 0) { code.resize(codeSize); }
        }
        check_error(err);
    }

    /**
    * Build an application hardware code.
    * @param applicationName [in] The application name.
    * @param certificate [in] The certificate.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The application hardware code.
    */
    static std::string getApplicationHardwareCode(const char *applicationName, const char *certificate) {
        std::string code;
        getApplicationHardwareCode(applicationName, certificate, code);
        return code;
    }

};
};