//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#pragma once
#include <id3BiosealCppWrapper/id3BiosealException.hpp>
#include <id3Bioseal/id3BiosealLogItem.h>

namespace id3BiosealCppWrapper
{

/**
 * Represents a log item.
 */
class LogItem
{
private:
    ID3_BIOSEAL_LOG_ITEM handle_{};

public:
    LogItem() {
        check_error(id3BiosealLogItem_Initialize(&handle_));
    }

    virtual ~LogItem() {
        if (handle_) {
            id3BiosealLogItem_Dispose(&handle_);
        }
    }


    LogItem(LogItem &&src) noexcept: handle_(src.handle_) {
        src.handle_ = nullptr;
    }

    ID3_BIOSEAL_LOG_ITEM getHandle() const { return handle_; }
    operator ID3_BIOSEAL_LOG_ITEM () const { return handle_; }

    // public getters and setters
    /**
    * Gets the the timestamp of the message.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The timestamp of the message.
    */
    long long getTimestamp() {
        long long timestamp;
        check_error(id3BiosealLogItem_GetTimestamp(handle_, &timestamp));
        return timestamp;
    }

    /**
    * Gets the the log message.
    * @param message [out] The log message.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The log message.
    */
    void getMessage(std::string & message) {
        int messageSize = -1;
        auto err = id3BiosealLogItem_GetMessage(handle_, message.data(), &messageSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            message.resize(messageSize);
            err = id3BiosealLogItem_GetMessage(handle_, message.data(), &messageSize);
            if (err == 0) { message.resize(messageSize); }
        }
        check_error(err);
    }

    /**
    * Gets the the log message.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The log message.
    */
    std::string getMessage() {
        std::string message;
        getMessage(message);
        return message;
    }

    /**
    * Gets the the log type.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The log type.
    */
    id3BiosealLogItemType getItemType() {
        id3BiosealLogItemType item_type;
        check_error(id3BiosealLogItem_GetItemType(handle_, &item_type));
        return item_type;
    }

    /**
    * Gets the the log item error code if any.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The log item error code if any.
    */
    id3BiosealError getErrorCode() {
        id3BiosealError error_code;
        check_error(id3BiosealLogItem_GetErrorCode(handle_, &error_code));
        return error_code;
    }

    // public methods
};
};