//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#pragma once
#include <id3BiosealCppWrapper/id3BiosealException.hpp>
#include <id3BiosealCppWrapper/id3BiosealLog.hpp>
#include <id3Bioseal/id3BiosealLotl.h>
#include "vector"

namespace id3BiosealCppWrapper
{

/**
 * Provides information about the LoTL (List of Trust List).
 */
class Lotl
{
private:
    ID3_BIOSEAL_LOTL handle_{};

public:
    Lotl() {
        check_error(id3BiosealLotl_Initialize(&handle_));
    }

    virtual ~Lotl() {
        if (handle_) {
            id3BiosealLotl_Dispose(&handle_);
        }
    }


    Lotl(Lotl &&src) noexcept: handle_(src.handle_) {
        src.handle_ = nullptr;
    }

    ID3_BIOSEAL_LOTL getHandle() const { return handle_; }
    operator ID3_BIOSEAL_LOTL () const { return handle_; }

    // public getters and setters
    /**
    * Gets the get the log.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval Get the log.
    */
    Log getLog() {
        Log log;
        check_error(id3BiosealLotl_GetLog(handle_, log));
        return log;
    }

    /**
    * Gets the true if the LoTL is a TLS.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval True if the LoTL is a TLS.
    */
    bool getIsTsl() {
        bool is_tsl;
        check_error(id3BiosealLotl_GetIsTsl(handle_, &is_tsl));
        return is_tsl;
    }

    // public methods
    /**
    * Returns the URL to the TSL or the next LoTL.
    * @param lotlData [in] The LotL data.
    * @param lotlDataSize [in] Size of the 'lotlData' buffer.
    * @param manifestId [in] The Manifest ID in ISO 22385 format (hexadecimal string).
    * @param iac [in] The IAC (optional).
    * @param url [out] The TSL url.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The TSL url.
    */
    void getUrl(unsigned char *lotlData, int lotlDataSize, const char *manifestId, const char *iac, std::string & url) {
        int urlSize = -1;
        auto err = id3BiosealLotl_GetUrl(handle_, lotlData, lotlDataSize, manifestId, iac, url.data(), &urlSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            url.resize(urlSize);
            err = id3BiosealLotl_GetUrl(handle_, lotlData, lotlDataSize, manifestId, iac, url.data(), &urlSize);
            if (err == 0) { url.resize(urlSize); }
        }
        check_error(err);
    }

    /**
    * Returns the URL to the TSL or the next LoTL.
    * @param lotlData [in] The LotL data.
    * @param lotlDataSize [in] Size of the 'lotlData' buffer.
    * @param manifestId [in] The Manifest ID in ISO 22385 format (hexadecimal string).
    * @param iac [in] The IAC (optional).
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The TSL url.
    */
    std::string getUrl(unsigned char *lotlData, int lotlDataSize, const char *manifestId, const char *iac) {
        std::string url;
        getUrl(lotlData, lotlDataSize, manifestId, iac, url);
        return url;
    }

    /**
    * Returns the URL to the manifest.
    * @param tslData [in] The TSL data.
    * @param tslDataSize [in] Size of the 'tslData' buffer.
    * @param url [out] The manifest url.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The manifest url.
    */
    void getManifestUrl(unsigned char *tslData, int tslDataSize, std::string & url) {
        int urlSize = -1;
        auto err = id3BiosealLotl_GetManifestUrl(handle_, tslData, tslDataSize, url.data(), &urlSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            url.resize(urlSize);
            err = id3BiosealLotl_GetManifestUrl(handle_, tslData, tslDataSize, url.data(), &urlSize);
            if (err == 0) { url.resize(urlSize); }
        }
        check_error(err);
    }

    /**
    * Returns the URL to the manifest.
    * @param tslData [in] The TSL data.
    * @param tslDataSize [in] Size of the 'tslData' buffer.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The manifest url.
    */
    std::string getManifestUrl(unsigned char *tslData, int tslDataSize) {
        std::string url;
        getManifestUrl(tslData, tslDataSize, url);
        return url;
    }

};
};