//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#pragma once
#include <id3BiosealCppWrapper/id3BiosealException.hpp>
#include <id3Bioseal/id3BiosealUserAuthentication.h>

namespace id3BiosealCppWrapper
{

/**
 * Represents the user authentication parameters.
 */
class UserAuthentication
{
private:
    ID3_BIOSEAL_USER_AUTHENTICATION handle_{};

public:
    UserAuthentication() {
        check_error(id3BiosealUserAuthentication_Initialize(&handle_));
    }

    virtual ~UserAuthentication() {
        if (handle_) {
            id3BiosealUserAuthentication_Dispose(&handle_);
        }
    }


    UserAuthentication(UserAuthentication &&src) noexcept: handle_(src.handle_) {
        src.handle_ = nullptr;
    }

    ID3_BIOSEAL_USER_AUTHENTICATION getHandle() const { return handle_; }
    operator ID3_BIOSEAL_USER_AUTHENTICATION () const { return handle_; }

    // public getters and setters
    /**
    * Gets the the user authentication method.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The user authentication method.
    */
    id3BiosealUserAuthenticationMethod getMethod() {
        id3BiosealUserAuthenticationMethod method;
        check_error(id3BiosealUserAuthentication_GetMethod(handle_, &method));
        return method;
    }

    /**
    * Gets the the name of the payload field containing the password value.
    * @param passwordFieldName [out] The name of the payload field containing the password value.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The name of the payload field containing the password value.
    */
    void getPasswordFieldName(std::string & passwordFieldName) {
        int passwordFieldNameSize = -1;
        auto err = id3BiosealUserAuthentication_GetPasswordFieldName(handle_, passwordFieldName.data(), &passwordFieldNameSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            passwordFieldName.resize(passwordFieldNameSize);
            err = id3BiosealUserAuthentication_GetPasswordFieldName(handle_, passwordFieldName.data(), &passwordFieldNameSize);
            if (err == 0) { passwordFieldName.resize(passwordFieldNameSize); }
        }
        check_error(err);
    }

    /**
    * Gets the the name of the payload field containing the password value.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The name of the payload field containing the password value.
    */
    std::string getPasswordFieldName() {
        std::string password_field_name;
        getPasswordFieldName(password_field_name);
        return password_field_name;
    }

    /**
    * Gets the the hash algorithm used to protect the password value. Possible values are: Sha256, Sha384, Sha512.
    * @param passwordHashAlgorithm [out] The hash algorithm used to protect the password value. Possible values are: Sha256, Sha384, Sha512.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The hash algorithm used to protect the password value. Possible values are: Sha256, Sha384, Sha512.
    */
    void getPasswordHashAlgorithm(std::string & passwordHashAlgorithm) {
        int passwordHashAlgorithmSize = -1;
        auto err = id3BiosealUserAuthentication_GetPasswordHashAlgorithm(handle_, passwordHashAlgorithm.data(), &passwordHashAlgorithmSize);
        if (err == (int)id3BiosealError_InsufficientBuffer) {
            passwordHashAlgorithm.resize(passwordHashAlgorithmSize);
            err = id3BiosealUserAuthentication_GetPasswordHashAlgorithm(handle_, passwordHashAlgorithm.data(), &passwordHashAlgorithmSize);
            if (err == 0) { passwordHashAlgorithm.resize(passwordHashAlgorithmSize); }
        }
        check_error(err);
    }

    /**
    * Gets the the hash algorithm used to protect the password value. Possible values are: Sha256, Sha384, Sha512.
    * @exception BiosealException An error has occured during Bioseal library execution.
    * @retval The hash algorithm used to protect the password value. Possible values are: Sha256, Sha384, Sha512.
    */
    std::string getPasswordHashAlgorithm() {
        std::string password_hash_algorithm;
        getPasswordHashAlgorithm(password_hash_algorithm);
        return password_hash_algorithm;
    }

    // public methods
};
};