//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DetectedDocument_h
#define _id3DetectedDocument_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentPointList.h>
#include <id3Document/id3DocumentInfo.h>
#include <id3Document/id3DocumentOrientation.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a detected document.
 */

/**
 * Handle to a DetectedDocument object of the Document module.
 */
typedef struct id3DetectedDocument *ID3_DETECTED_DOCUMENT;

/**
 * @brief Initializes a DetectedDocument object.
 * @param phDetectedDocument [in] Handle to the DetectedDocument object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DetectedDocument_Initialize(ID3_DETECTED_DOCUMENT *phDetectedDocument);

/**
 * @brief Releases the memory allocated for a DetectedDocument.
 * @param phDetectedDocument [in] Handle to the DetectedDocument object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DetectedDocument_Dispose(ID3_DETECTED_DOCUMENT *phDetectedDocument);

/**
 * @brief Checks if the handle object is valid.
 * @param hDetectedDocument [in] Handle to the DetectedDocument object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DetectedDocument_CheckHandle(ID3_DETECTED_DOCUMENT hDetectedDocument);

/**
 * @brief Copies an instance of DetectedDocument object to another.
 * @param hDetectedDocumentSrc [in] Handle to the source DetectedDocument object.
 * @param hDetectedDocumentDst [in] Handle to the destination DetectedDocument object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DetectedDocument_CopyTo(ID3_DETECTED_DOCUMENT hDetectedDocumentSrc, ID3_DETECTED_DOCUMENT hDetectedDocumentDst);

/**
 * Gets the boundaries of the document detected, in the following order: top-left, top-right, bottom-right, bottom-left.
 * @param hDetectedDocument [in] Handle to the DetectedDocument object.
 * @param hBounds [out] Boundaries of the document detected, in the following order: top-left, top-right, bottom-right, bottom-left.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DetectedDocument_GetBounds(ID3_DETECTED_DOCUMENT hDetectedDocument, ID3_DOCUMENT_POINT_LIST hBounds);

/**
 * Sets the boundaries of the document detected, in the following order: top-left, top-right, bottom-right, bottom-left.
 * @param hDetectedDocument [in] Handle to the DetectedDocument object.
 * @param hBounds [in] Boundaries of the document detected, in the following order: top-left, top-right, bottom-right, bottom-left.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DetectedDocument_SetBounds(ID3_DETECTED_DOCUMENT hDetectedDocument, ID3_DOCUMENT_POINT_LIST hBounds);

/**
 * Gets the confidence score of the detected document, from 0 to 100.
 * @param hDetectedDocument [in] Handle to the DetectedDocument object.
 * @param confidence [out] Confidence score of the detected document, from 0 to 100.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DetectedDocument_GetConfidence(ID3_DETECTED_DOCUMENT hDetectedDocument, int *confidence);

/**
 * Gets the document information.
 * @param hDetectedDocument [in] Handle to the DetectedDocument object.
 * @param hInfo [out] Document information.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DetectedDocument_GetInfo(ID3_DETECTED_DOCUMENT hDetectedDocument, ID3_DOCUMENT_INFO hInfo);

/**
 * Gets the document orientation in source image. [0,0,0] means the document is in front of the camera.
 * @param hDetectedDocument [in] Handle to the DetectedDocument object.
 * @param sOrientation [out] Document orientation in source image. [0,0,0] means the document is in front of the camera.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DetectedDocument_GetOrientation(ID3_DETECTED_DOCUMENT hDetectedDocument, id3DocumentOrientation *sOrientation);

#ifdef __cplusplus
}
#endif

#endif
