//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentStringArray_h
#define _id3DocumentStringArray_h

#include <id3Document/id3DocumentExports.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Handle to a StringArray object of the Document module.
 */
typedef struct id3DocumentStringArray *ID3_DOCUMENT_STRING_ARRAY;

/**
 * @brief Initializes a StringArray object.
 * @param phStringArray [in] Handle to the StringArray object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringArray_Initialize(ID3_DOCUMENT_STRING_ARRAY *phStringArray);

/**
 * @brief Releases the memory allocated for a StringArray.
 * @param phStringArray [in] Handle to the StringArray object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringArray_Dispose(ID3_DOCUMENT_STRING_ARRAY *phStringArray);

/**
 * @brief Checks if the handle object is valid.
 * @param hStringArray [in] Handle to the StringArray object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringArray_CheckHandle(ID3_DOCUMENT_STRING_ARRAY hStringArray);

/**
 * @brief Copies an instance of StringArray object to another.
 * @param hStringArraySrc [in] Handle to the source StringArray object.
 * @param hStringArrayDst [in] Handle to the destination StringArray object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringArray_CopyTo(ID3_DOCUMENT_STRING_ARRAY hStringArraySrc, ID3_DOCUMENT_STRING_ARRAY hStringArrayDst);

/**
 * Clears the StringArray object.
 * @param hStringArray [in] Handle to the StringArray object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringArray_Clear(ID3_DOCUMENT_STRING_ARRAY hStringArray);

/**
 * Adds an item to the StringArray object.
 * @param hStringArray [in] Handle to the StringArray object.
 * @param item [in]  item to add.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringArray_Add(ID3_DOCUMENT_STRING_ARRAY hStringArray, const char *item);

/**
 * Gets an item of the StringArray object.
 * @param hStringArray [in] Handle to the StringArray object.
 * @param index [in] Index of the  item to get.
 * @param item [out]  item to get.
 * @param itemSize [in,out] Size of the 'item' buffer. Receives the minimum required buffer size to hold the 'item' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'item' (excluding terminating NULL character).
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringArray_Get(ID3_DOCUMENT_STRING_ARRAY hStringArray, int index, char *item, int *itemSize);

/**
 * Sets an item of the StringArray object.
 * @param hStringArray [in] Handle to the StringArray object.
 * @param index [in] Index of the  item to set.
 * @param item [in]  item to set.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringArray_Set(ID3_DOCUMENT_STRING_ARRAY hStringArray, int index, const char *item);

/**
 * Gets the number of elements in the StringArray object.
 * @param hStringArray [in] Handle to the StringArray object.
 * @param count [out] Number of elements in the  object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringArray_GetCount(ID3_DOCUMENT_STRING_ARRAY hStringArray, int *count);

/**
 * Removes an element of the StringArray object.
 * @param hStringArray [in] Handle to the StringArray object.
 * @param index [in] Index of the  item to remove.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringArray_RemoveAt(ID3_DOCUMENT_STRING_ARRAY hStringArray, int index);

/**
 * StringArray object.
 * @param hStringArray [in] Handle to the StringArray object.
 * @param newSize [in]  object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringArray_Resize(ID3_DOCUMENT_STRING_ARRAY hStringArray, int newSize);

#ifdef __cplusplus
}
#endif

#endif
