//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentAuthenticator_h
#define _id3DocumentAuthenticator_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentImage.h>
#include <id3Document/id3DocumentHologramAuthenticationResultList.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Verifies the authenticity of a document using security features such as holograms.
 */

/**
 * Handle to a DocumentAuthenticator object of the Document module.
 */
typedef struct id3DocumentAuthenticator *ID3_DOCUMENT_AUTHENTICATOR;

/**
 * @brief Initializes a DocumentAuthenticator object.
 * @param phDocumentAuthenticator [in] Handle to the DocumentAuthenticator object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentAuthenticator_Initialize(ID3_DOCUMENT_AUTHENTICATOR *phDocumentAuthenticator);

/**
 * @brief Releases the memory allocated for a DocumentAuthenticator.
 * @param phDocumentAuthenticator [in] Handle to the DocumentAuthenticator object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentAuthenticator_Dispose(ID3_DOCUMENT_AUTHENTICATOR *phDocumentAuthenticator);

/**
 * @brief Checks if the handle object is valid.
 * @param hDocumentAuthenticator [in] Handle to the DocumentAuthenticator object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentAuthenticator_CheckHandle(ID3_DOCUMENT_AUTHENTICATOR hDocumentAuthenticator);

/**
 * Gets the document type to authenticate.
 * @param hDocumentAuthenticator [in] Handle to the DocumentAuthenticator object.
 * @param documentName [out] Document type to authenticate.
 * @param documentNameSize [in,out] Size of the 'documentName' buffer. Receives the minimum required buffer size to hold the 'documentName' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'documentName' (excluding terminating NULL character).
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentAuthenticator_GetDocumentName(ID3_DOCUMENT_AUTHENTICATOR hDocumentAuthenticator, char *documentName, int *documentNameSize);

/**
 * Sets the document type to authenticate.
 * @param hDocumentAuthenticator [in] Handle to the DocumentAuthenticator object.
 * @param documentName [in] Document type to authenticate.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentAuthenticator_SetDocumentName(ID3_DOCUMENT_AUTHENTICATOR hDocumentAuthenticator, const char *documentName);

/**
 * Gets the size of the FIFO frame buffer, used for processing.
 * @param hDocumentAuthenticator [in] Handle to the DocumentAuthenticator object.
 * @param temporalWindowSize [out] Size of the FIFO frame buffer, used for processing.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentAuthenticator_GetTemporalWindowSize(ID3_DOCUMENT_AUTHENTICATOR hDocumentAuthenticator, int *temporalWindowSize);

/**
 * Sets the size of the FIFO frame buffer, used for processing.
 * @param hDocumentAuthenticator [in] Handle to the DocumentAuthenticator object.
 * @param temporalWindowSize [in] Size of the FIFO frame buffer, used for processing.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentAuthenticator_SetTemporalWindowSize(ID3_DOCUMENT_AUTHENTICATOR hDocumentAuthenticator, int temporalWindowSize);

/**
 * Appends a frame in the FIFO buffer.
 * @param hDocumentAuthenticator [in] Handle to the DocumentAuthenticator object.
 * @param hImage [in] Source image of the aligned document to append
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentAuthenticator_AppendFrame(ID3_DOCUMENT_AUTHENTICATOR hDocumentAuthenticator, ID3_DOCUMENT_IMAGE hImage);

/**
 * Authenticates the holograms on the current frame buffer.
 * Important: Buffer must be full to the temporal window size.
 * @param hDocumentAuthenticator [in] Handle to the DocumentAuthenticator object.
 * @param hScoreList [out] List of result for each hologram enrolled in document.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentAuthenticator_AuthenticateHolograms(ID3_DOCUMENT_AUTHENTICATOR hDocumentAuthenticator, ID3_DOCUMENT_HOLOGRAM_AUTHENTICATION_RESULT_LIST hScoreList);

/**
 * Checks if the frame buffer is full and ready to authenticate for all holograms.
 * @param hDocumentAuthenticator [in] Handle to the DocumentAuthenticator object.
 * @param result [out] A value indicating whether the frame buffer is full.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentAuthenticator_IsFull(ID3_DOCUMENT_AUTHENTICATOR hDocumentAuthenticator, bool *result);

#ifdef __cplusplus
}
#endif

#endif
