//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentInfoDictionary_h
#define _id3DocumentInfoDictionary_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentStringList.h>
#include <id3Document/id3DocumentInfo.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a dictionary of document information.
 */

/**
 * Handle to a DocumentInfoDictionary object of the Document module.
 */
typedef struct id3DocumentInfoDictionary *ID3_DOCUMENT_INFO_DICTIONARY;

/**
 * @brief Initializes a DocumentInfoDictionary object.
 * @param phDocumentInfoDictionary [in] Handle to the DocumentInfoDictionary object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfoDictionary_Initialize(ID3_DOCUMENT_INFO_DICTIONARY *phDocumentInfoDictionary);

/**
 * @brief Releases the memory allocated for a DocumentInfoDictionary.
 * @param phDocumentInfoDictionary [in] Handle to the DocumentInfoDictionary object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfoDictionary_Dispose(ID3_DOCUMENT_INFO_DICTIONARY *phDocumentInfoDictionary);

/**
 * @brief Checks if the handle object is valid.
 * @param hDocumentInfoDictionary [in] Handle to the DocumentInfoDictionary object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfoDictionary_CheckHandle(ID3_DOCUMENT_INFO_DICTIONARY hDocumentInfoDictionary);

/**
 * @brief Copies an instance of DocumentInfoDictionary object to another.
 * @param hDocumentInfoDictionarySrc [in] Handle to the source DocumentInfoDictionary object.
 * @param hDocumentInfoDictionaryDst [in] Handle to the destination DocumentInfoDictionary object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfoDictionary_CopyTo(ID3_DOCUMENT_INFO_DICTIONARY hDocumentInfoDictionarySrc, ID3_DOCUMENT_INFO_DICTIONARY hDocumentInfoDictionaryDst);

/**
 * Gets the gets a string list containing the keys in the dict.
 * @param hDocumentInfoDictionary [in] Handle to the DocumentInfoDictionary object.
 * @param hKeys [out] Gets a string list containing the keys in the dict.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfoDictionary_GetKeys(ID3_DOCUMENT_INFO_DICTIONARY hDocumentInfoDictionary, ID3_DOCUMENT_STRING_LIST hKeys);

/**
 * Clears the DocumentInfoDictionary object.
 * @param hDocumentInfoDictionary [in] Handle to the DocumentInfoDictionary object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfoDictionary_Clear(ID3_DOCUMENT_INFO_DICTIONARY hDocumentInfoDictionary);

/**
 * Adds an item to the DocumentInfoDictionary object.
 * @param hDocumentInfoDictionary [in] Handle to the DocumentInfoDictionary object.
 * @param key [in] Unique key of the DocumentInfo item to add.
 * @param hDocumentInfoItem [in] DocumentInfo item to add.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfoDictionary_Add(ID3_DOCUMENT_INFO_DICTIONARY hDocumentInfoDictionary, const char *key, ID3_DOCUMENT_INFO hDocumentInfoItem);

/**
 * Gets an item of the DocumentInfoDictionary object.
 * @param hDocumentInfoDictionary [in] Handle to the DocumentInfoDictionary object.
 * @param key [in] Unique key of the DocumentInfo item to get.
 * @param hDocumentInfoItem [out] DocumentInfo item to get.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfoDictionary_Get(ID3_DOCUMENT_INFO_DICTIONARY hDocumentInfoDictionary, const char *key, ID3_DOCUMENT_INFO hDocumentInfoItem);

/**
 * Sets an item of the DocumentInfoDictionary object.
 * @param hDocumentInfoDictionary [in] Handle to the DocumentInfoDictionary object.
 * @param key [in] Unique key of the DocumentInfo item to set.
 * @param hDocumentInfoItem [in] DocumentInfo item to set.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfoDictionary_Set(ID3_DOCUMENT_INFO_DICTIONARY hDocumentInfoDictionary, const char *key, ID3_DOCUMENT_INFO hDocumentInfoItem);

/**
 * Gets the number of elements in the DocumentInfoDictionary object.
 * @param hDocumentInfoDictionary [in] Handle to the DocumentInfoDictionary object.
 * @param count [out] Number of elements in the DocumentInfo object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfoDictionary_GetCount(ID3_DOCUMENT_INFO_DICTIONARY hDocumentInfoDictionary, int *count);

/**
 * Determines whether the DocumentInfoDictionaryobject contains the specified key.
 * @param hDocumentInfoDictionary [in] Handle to the DocumentInfoDictionary object.
 * @param key [in] The key to locate in the DocumentInfo object.
 * @param result [out] true if the DocumentInfo object contains an element with the specified key, otherwise false.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfoDictionary_ContainsKey(ID3_DOCUMENT_INFO_DICTIONARY hDocumentInfoDictionary, const char *key, bool *result);

/**
 * Removes an element of the DocumentInfoDictionary object.
 * @param hDocumentInfoDictionary [in] Handle to the DocumentInfoDictionary object.
 * @param key [in] Unique key of the DocumentInfo item to remove.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfoDictionary_Remove(ID3_DOCUMENT_INFO_DICTIONARY hDocumentInfoDictionary, const char *key);

#ifdef __cplusplus
}
#endif

#endif
