//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentMrzReader_h
#define _id3DocumentMrzReader_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentModel.h>
#include <id3Document/id3DocumentProcessingUnit.h>
#include <id3Document/id3DocumentImage.h>
#include <id3Document/id3DocumentMrzReadingResult.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Detects and reads an MRZ in an image.
 */

/**
 * Handle to a MrzReader object of the Document module.
 */
typedef struct id3DocumentMrzReader *ID3_DOCUMENT_MRZ_READER;

/**
 * @brief Initializes a MrzReader object.
 * @param phMrzReader [in] Handle to the MrzReader object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReader_Initialize(ID3_DOCUMENT_MRZ_READER *phMrzReader);

/**
 * @brief Releases the memory allocated for a MrzReader.
 * @param phMrzReader [in] Handle to the MrzReader object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReader_Dispose(ID3_DOCUMENT_MRZ_READER *phMrzReader);

/**
 * @brief Checks if the handle object is valid.
 * @param hMrzReader [in] Handle to the MrzReader object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReader_CheckHandle(ID3_DOCUMENT_MRZ_READER hMrzReader);

/**
 * Gets the selection of the binarization method internally used in the algorithm.
 * @param hMrzReader [in] Handle to the MrzReader object.
 * @param binarizationMethod [out] Selection of the binarization method internally used in the algorithm.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReader_GetBinarizationMethod(ID3_DOCUMENT_MRZ_READER hMrzReader, int *binarizationMethod);

/**
 * Sets the selection of the binarization method internally used in the algorithm.
 * @param hMrzReader [in] Handle to the MrzReader object.
 * @param binarizationMethod [in] Selection of the binarization method internally used in the algorithm.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReader_SetBinarizationMethod(ID3_DOCUMENT_MRZ_READER hMrzReader, int binarizationMethod);

/**
 * Gets the AI model used to detect the MRZ.
 * Hint: Default value is MrzDetector2A.
 * @param hMrzReader [in] Handle to the MrzReader object.
 * @param eModel [out] AI model used to detect the MRZ.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReader_GetModel(ID3_DOCUMENT_MRZ_READER hMrzReader, id3DocumentModel *eModel);

/**
 * Sets the AI model used to detect the MRZ.
 * Hint: Default value is MrzDetector2A.
 * @param hMrzReader [in] Handle to the MrzReader object.
 * @param eModel [in] AI model used to detect the MRZ.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReader_SetModel(ID3_DOCUMENT_MRZ_READER hMrzReader, id3DocumentModel eModel);

/**
 * Gets the processing unit where to run the detection process.
 * Hint: Default value is CPU.
 * @param hMrzReader [in] Handle to the MrzReader object.
 * @param eProcessingUnit [out] Processing unit where to run the detection process.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReader_GetProcessingUnit(ID3_DOCUMENT_MRZ_READER hMrzReader, id3DocumentProcessingUnit *eProcessingUnit);

/**
 * Sets the processing unit where to run the detection process.
 * Hint: Default value is CPU.
 * @param hMrzReader [in] Handle to the MrzReader object.
 * @param eProcessingUnit [in] Processing unit where to run the detection process.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReader_SetProcessingUnit(ID3_DOCUMENT_MRZ_READER hMrzReader, id3DocumentProcessingUnit eProcessingUnit);

/**
 * Gets the number of threads to use for reading.
 * Hint: Default value is 1. Allocating more than 1 thread here can increase the speed of the process.
 * @param hMrzReader [in] Handle to the MrzReader object.
 * @param threadCount [out] Number of threads to use for reading.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReader_GetThreadCount(ID3_DOCUMENT_MRZ_READER hMrzReader, int *threadCount);

/**
 * Sets the number of threads to use for reading.
 * Hint: Default value is 1. Allocating more than 1 thread here can increase the speed of the process.
 * @param hMrzReader [in] Handle to the MrzReader object.
 * @param threadCount [in] Number of threads to use for reading.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReader_SetThreadCount(ID3_DOCUMENT_MRZ_READER hMrzReader, int threadCount);

/**
 * Detects and reads an MRZ in an image.
 * @param hMrzReader [in] Handle to the MrzReader object.
 * @param hImage [in] Source image to process
 * @param hResult [out] MRZ reading output.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReader_ReadMrz(ID3_DOCUMENT_MRZ_READER hMrzReader, ID3_DOCUMENT_IMAGE hImage, ID3_DOCUMENT_MRZ_READING_RESULT hResult);

#ifdef __cplusplus
}
#endif

#endif
