//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentPointList_h
#define _id3DocumentPointList_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentPoint.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a point list.
 */

/**
 * Handle to a PointList object of the Document module.
 */
typedef struct id3DocumentPointList *ID3_DOCUMENT_POINT_LIST;

/**
 * @brief Initializes a PointList object.
 * @param phPointList [in] Handle to the PointList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentPointList_Initialize(ID3_DOCUMENT_POINT_LIST *phPointList);

/**
 * @brief Releases the memory allocated for a PointList.
 * @param phPointList [in] Handle to the PointList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentPointList_Dispose(ID3_DOCUMENT_POINT_LIST *phPointList);

/**
 * @brief Checks if the handle object is valid.
 * @param hPointList [in] Handle to the PointList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentPointList_CheckHandle(ID3_DOCUMENT_POINT_LIST hPointList);

/**
 * @brief Copies an instance of PointList object to another.
 * @param hPointListSrc [in] Handle to the source PointList object.
 * @param hPointListDst [in] Handle to the destination PointList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentPointList_CopyTo(ID3_DOCUMENT_POINT_LIST hPointListSrc, ID3_DOCUMENT_POINT_LIST hPointListDst);

/**
 * Clears the PointList object.
 * @param hPointList [in] Handle to the PointList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentPointList_Clear(ID3_DOCUMENT_POINT_LIST hPointList);

/**
 * Adds an item to the PointList object.
 * @param hPointList [in] Handle to the PointList object.
 * @param sPointItem [in] Point item to add.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentPointList_Add(ID3_DOCUMENT_POINT_LIST hPointList, const id3DocumentPoint *sPointItem);

/**
 * Gets an item of the PointList object.
 * @param hPointList [in] Handle to the PointList object.
 * @param index [in] Index of the Point item to get.
 * @param sPointItem [out] Point item to get.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentPointList_Get(ID3_DOCUMENT_POINT_LIST hPointList, int index, id3DocumentPoint *sPointItem);

/**
 * Sets an item of the PointList object.
 * @param hPointList [in] Handle to the PointList object.
 * @param index [in] Index of the Point item to set.
 * @param sPointItem [in] Point item to set.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentPointList_Set(ID3_DOCUMENT_POINT_LIST hPointList, int index, const id3DocumentPoint *sPointItem);

/**
 * Gets the number of elements in the PointList object.
 * @param hPointList [in] Handle to the PointList object.
 * @param count [out] Number of elements in the Point object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentPointList_GetCount(ID3_DOCUMENT_POINT_LIST hPointList, int *count);

/**
 * Removes an element of the PointList object.
 * @param hPointList [in] Handle to the PointList object.
 * @param index [in] Index of the Point item to remove.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentPointList_RemoveAt(ID3_DOCUMENT_POINT_LIST hPointList, int index);

/**
 * PointList object.
 * @param hPointList [in] Handle to the PointList object.
 * @param newSize [in] Point object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentPointList_Resize(ID3_DOCUMENT_POINT_LIST hPointList, int newSize);

#ifdef __cplusplus
}
#endif

#endif
