//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentDetector_h
#define _id3DocumentDetector_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentModel.h>
#include <id3Document/id3DocumentProcessingUnit.h>
#include <id3Document/id3DocumentImage.h>
#include <id3Document/id3DetectedDocument.h>
#include <id3Document/id3DocumentRectangle.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Detects documents in images.
 */

/**
 * Handle to a DocumentDetector object of the Document module.
 */
typedef struct id3DocumentDetector *ID3_DOCUMENT_DETECTOR;

/**
 * @brief Initializes a DocumentDetector object.
 * @param phDocumentDetector [in] Handle to the DocumentDetector object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_Initialize(ID3_DOCUMENT_DETECTOR *phDocumentDetector);

/**
 * @brief Releases the memory allocated for a DocumentDetector.
 * @param phDocumentDetector [in] Handle to the DocumentDetector object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_Dispose(ID3_DOCUMENT_DETECTOR *phDocumentDetector);

/**
 * @brief Checks if the handle object is valid.
 * @param hDocumentDetector [in] Handle to the DocumentDetector object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_CheckHandle(ID3_DOCUMENT_DETECTOR hDocumentDetector);

/**
 * Gets the confidence threshold, from 0 to 100+.
 * Hint: Default value is 50. Lower threshold can increase number of false detection.
 * @param hDocumentDetector [in] Handle to the DocumentDetector object.
 * @param confidenceThreshold [out] Confidence threshold, from 0 to 100+.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_GetConfidenceThreshold(ID3_DOCUMENT_DETECTOR hDocumentDetector, int *confidenceThreshold);

/**
 * Sets the confidence threshold, from 0 to 100+.
 * Hint: Default value is 50. Lower threshold can increase number of false detection.
 * @param hDocumentDetector [in] Handle to the DocumentDetector object.
 * @param confidenceThreshold [in] Confidence threshold, from 0 to 100+.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_SetConfidenceThreshold(ID3_DOCUMENT_DETECTOR hDocumentDetector, int confidenceThreshold);

/**
 * Gets the ratio of the estimated width of the searched document in the image.
 * Hint: Default value is 1.
 * @param hDocumentDetector [in] Handle to the DocumentDetector object.
 * @param documentWidthRatio [out] Ratio of the estimated width of the searched document in the image.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_GetDocumentWidthRatio(ID3_DOCUMENT_DETECTOR hDocumentDetector, float *documentWidthRatio);

/**
 * Sets the ratio of the estimated width of the searched document in the image.
 * Hint: Default value is 1.
 * @param hDocumentDetector [in] Handle to the DocumentDetector object.
 * @param documentWidthRatio [in] Ratio of the estimated width of the searched document in the image.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_SetDocumentWidthRatio(ID3_DOCUMENT_DETECTOR hDocumentDetector, float documentWidthRatio);

/**
 * Gets the model to be used for document detection.
 * Hint: Default value is DocumentDetector2A.
 * @param hDocumentDetector [in] Handle to the DocumentDetector object.
 * @param eModel [out] Model to be used for document detection.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_GetModel(ID3_DOCUMENT_DETECTOR hDocumentDetector, id3DocumentModel *eModel);

/**
 * Sets the model to be used for document detection.
 * Hint: Default value is DocumentDetector2A.
 * @param hDocumentDetector [in] Handle to the DocumentDetector object.
 * @param eModel [in] Model to be used for document detection.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_SetModel(ID3_DOCUMENT_DETECTOR hDocumentDetector, id3DocumentModel eModel);

/**
 * Gets the processing unit where to run the detection process.
 * Hint: Default value is CPU.
 * @param hDocumentDetector [in] Handle to the DocumentDetector object.
 * @param eProcessingUnit [out] Processing unit where to run the detection process.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_GetProcessingUnit(ID3_DOCUMENT_DETECTOR hDocumentDetector, id3DocumentProcessingUnit *eProcessingUnit);

/**
 * Sets the processing unit where to run the detection process.
 * Hint: Default value is CPU.
 * @param hDocumentDetector [in] Handle to the DocumentDetector object.
 * @param eProcessingUnit [in] Processing unit where to run the detection process.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_SetProcessingUnit(ID3_DOCUMENT_DETECTOR hDocumentDetector, id3DocumentProcessingUnit eProcessingUnit);

/**
 * Gets the number of threads to use for detection.
 * Hint: Default value is 1. Allocating more than 1 thread here can increase the speed of the process.
 * @param hDocumentDetector [in] Handle to the DocumentDetector object.
 * @param threadCount [out] Number of threads to use for detection.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_GetThreadCount(ID3_DOCUMENT_DETECTOR hDocumentDetector, int *threadCount);

/**
 * Sets the number of threads to use for detection.
 * Hint: Default value is 1. Allocating more than 1 thread here can increase the speed of the process.
 * @param hDocumentDetector [in] Handle to the DocumentDetector object.
 * @param threadCount [in] Number of threads to use for detection.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_SetThreadCount(ID3_DOCUMENT_DETECTOR hDocumentDetector, int threadCount);

/**
 * Aligns and crops the specified document on the specified image and returns a new DocumentImage.
 * @param hDocumentDetector [in] Handle to the DocumentDetector object.
 * @param hSourceImage [in] The source image containing the document.
 * @param hDetectedDocument [in] The detected document in the image.
 * @param hAlignedImage [out] Realigned and cropped image of the document.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_AlignDocument(ID3_DOCUMENT_DETECTOR hDocumentDetector, ID3_DOCUMENT_IMAGE hSourceImage, ID3_DETECTED_DOCUMENT hDetectedDocument, ID3_DOCUMENT_IMAGE hAlignedImage);

/**
 * Detects a document on a delimited area of the specified DocumentImage and returns a DetectedDocument.
 * Important: Document templates must be loaded before calling this method.
 * @param hDocumentDetector [in] Handle to the DocumentDetector object.
 * @param hImage [in] Source image to process
 * @param sDetectionZone [in] Crop in the image where to look for document. Default is the full image. Output corners are in the full image referential.
 * @param hDetectedDocument [out] Detected document in the image
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_DetectDocument(ID3_DOCUMENT_DETECTOR hDocumentDetector, ID3_DOCUMENT_IMAGE hImage, const id3DocumentRectangle *sDetectionZone, ID3_DETECTED_DOCUMENT hDetectedDocument);

/**
 * Detects a document with specified name on a delimited area of the specified DocumentImage and returns a DetectedDocument.
 * Important: The relevant document template must be loaded before calling this method.
 * @param hDocumentDetector [in] Handle to the DocumentDetector object.
 * @param hImage [in] The source image to process.
 * @param documentName [in] The name of the document to be searched for. Must be a key loaded in DocumentLibrary.
 * @param sDetectionZone [in] A delimited area in the image where to search for the document. Default is the full image. Output corners are in the full image referential.
 * @param hDetectedDocument [out] Detected document in the image
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentDetector_DetectDocumentByName(ID3_DOCUMENT_DETECTOR hDocumentDetector, ID3_DOCUMENT_IMAGE hImage, const char *documentName, const id3DocumentRectangle *sDetectionZone, ID3_DETECTED_DOCUMENT hDetectedDocument);

#ifdef __cplusplus
}
#endif

#endif
