//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentInfo_h
#define _id3DocumentInfo_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentCategory.h>
#include <id3Document/id3DocumentFormat.h>
#include <id3Document/id3DocumentMrzType.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents document information.
 */

/**
 * Handle to a DocumentInfo object of the Document module.
 */
typedef struct id3DocumentInfo *ID3_DOCUMENT_INFO;

/**
 * @brief Initializes a DocumentInfo object.
 * @param phDocumentInfo [in] Handle to the DocumentInfo object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfo_Initialize(ID3_DOCUMENT_INFO *phDocumentInfo);

/**
 * @brief Releases the memory allocated for a DocumentInfo.
 * @param phDocumentInfo [in] Handle to the DocumentInfo object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfo_Dispose(ID3_DOCUMENT_INFO *phDocumentInfo);

/**
 * @brief Checks if the handle object is valid.
 * @param hDocumentInfo [in] Handle to the DocumentInfo object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfo_CheckHandle(ID3_DOCUMENT_INFO hDocumentInfo);

/**
 * @brief Copies an instance of DocumentInfo object to another.
 * @param hDocumentInfoSrc [in] Handle to the source DocumentInfo object.
 * @param hDocumentInfoDst [in] Handle to the destination DocumentInfo object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfo_CopyTo(ID3_DOCUMENT_INFO hDocumentInfoSrc, ID3_DOCUMENT_INFO hDocumentInfoDst);

/**
 * Gets the the document category.
 * @param hDocumentInfo [in] Handle to the DocumentInfo object.
 * @param eCategory [out] The document category.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfo_GetCategory(ID3_DOCUMENT_INFO hDocumentInfo, id3DocumentCategory *eCategory);

/**
 * Gets the the ISO 3166-1 alpha-3 code identifying the country issuing the document.
 * @param hDocumentInfo [in] Handle to the DocumentInfo object.
 * @param country [out] The ISO 3166-1 alpha-3 code identifying the country issuing the document.
 * @param countrySize [in,out] Size of the 'country' buffer. Receives the minimum required buffer size to hold the 'country' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'country' (excluding terminating NULL character).
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfo_GetCountry(ID3_DOCUMENT_INFO hDocumentInfo, char *country, int *countrySize);

/**
 * Gets the the document's date of first issue, in the form `YYYY-MM-DD`.
 * @param hDocumentInfo [in] Handle to the DocumentInfo object.
 * @param date [out] The document's date of first issue, in the form `YYYY-MM-DD`.
 * @param dateSize [in,out] Size of the 'date' buffer. Receives the minimum required buffer size to hold the 'date' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'date' (excluding terminating NULL character).
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfo_GetDate(ID3_DOCUMENT_INFO hDocumentInfo, char *date, int *dateSize);

/**
 * Gets the the document description.
 * @param hDocumentInfo [in] Handle to the DocumentInfo object.
 * @param description [out] The document description.
 * @param descriptionSize [in,out] Size of the 'description' buffer. Receives the minimum required buffer size to hold the 'description' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'description' (excluding terminating NULL character).
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfo_GetDescription(ID3_DOCUMENT_INFO hDocumentInfo, char *description, int *descriptionSize);

/**
 * Gets the the document format.
 * @param hDocumentInfo [in] Handle to the DocumentInfo object.
 * @param eFormat [out] The document format.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfo_GetFormat(ID3_DOCUMENT_INFO hDocumentInfo, id3DocumentFormat *eFormat);

/**
 * Gets the the document identifier.
 * @param hDocumentInfo [in] Handle to the DocumentInfo object.
 * @param identifier [out] The document identifier.
 * @param identifierSize [in,out] Size of the 'identifier' buffer. Receives the minimum required buffer size to hold the 'identifier' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'identifier' (excluding terminating NULL character).
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfo_GetIdentifier(ID3_DOCUMENT_INFO hDocumentInfo, char *identifier, int *identifierSize);

/**
 * Gets the the type of machine-readable zone (MRZ), if applicable.
 * @param hDocumentInfo [in] Handle to the DocumentInfo object.
 * @param eMrzType [out] The type of machine-readable zone (MRZ), if applicable.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfo_GetMrzType(ID3_DOCUMENT_INFO hDocumentInfo, id3DocumentMrzType *eMrzType);

/**
 * Gets the the document type name.
 * @param hDocumentInfo [in] Handle to the DocumentInfo object.
 * @param name [out] The document type name.
 * @param nameSize [in,out] Size of the 'name' buffer. Receives the minimum required buffer size to hold the 'name' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'name' (excluding terminating NULL character).
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfo_GetName(ID3_DOCUMENT_INFO hDocumentInfo, char *name, int *nameSize);

/**
 * Gets the the page number of the document.
 * @param hDocumentInfo [in] Handle to the DocumentInfo object.
 * @param pageNumber [out] The page number of the document.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentInfo_GetPageNumber(ID3_DOCUMENT_INFO hDocumentInfo, int *pageNumber);

#ifdef __cplusplus
}
#endif

#endif
