//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentStringDict_h
#define _id3DocumentStringDict_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentStringList.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a string dictionary.
 */

/**
 * Handle to a StringDict object of the Document module.
 */
typedef struct id3DocumentStringDict *ID3_DOCUMENT_STRING_DICT;

/**
 * @brief Initializes a StringDict object.
 * @param phStringDict [in] Handle to the StringDict object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringDict_Initialize(ID3_DOCUMENT_STRING_DICT *phStringDict);

/**
 * @brief Releases the memory allocated for a StringDict.
 * @param phStringDict [in] Handle to the StringDict object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringDict_Dispose(ID3_DOCUMENT_STRING_DICT *phStringDict);

/**
 * @brief Checks if the handle object is valid.
 * @param hStringDict [in] Handle to the StringDict object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringDict_CheckHandle(ID3_DOCUMENT_STRING_DICT hStringDict);

/**
 * @brief Copies an instance of StringDict object to another.
 * @param hStringDictSrc [in] Handle to the source StringDict object.
 * @param hStringDictDst [in] Handle to the destination StringDict object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringDict_CopyTo(ID3_DOCUMENT_STRING_DICT hStringDictSrc, ID3_DOCUMENT_STRING_DICT hStringDictDst);

/**
 * Gets the gets a string list containing the keys in the dict.
 * @param hStringDict [in] Handle to the StringDict object.
 * @param hKeys [out] Gets a string list containing the keys in the dict.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringDict_GetKeys(ID3_DOCUMENT_STRING_DICT hStringDict, ID3_DOCUMENT_STRING_LIST hKeys);

/**
 * Clears the StringDict object.
 * @param hStringDict [in] Handle to the StringDict object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringDict_Clear(ID3_DOCUMENT_STRING_DICT hStringDict);

/**
 * Adds an item to the StringDict object.
 * @param hStringDict [in] Handle to the StringDict object.
 * @param key [in] Unique key of the  item to add.
 * @param item [in]  item to add.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringDict_Add(ID3_DOCUMENT_STRING_DICT hStringDict, const char *key, const char *item);

/**
 * Gets an item of the StringDict object.
 * @param hStringDict [in] Handle to the StringDict object.
 * @param key [in] Unique key of the  item to get.
 * @param item [out]  item to get.
 * @param itemSize [in,out] Size of the 'item' buffer. Receives the minimum required buffer size to hold the 'item' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'item' (excluding terminating NULL character).
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringDict_Get(ID3_DOCUMENT_STRING_DICT hStringDict, const char *key, char *item, int *itemSize);

/**
 * Sets an item of the StringDict object.
 * @param hStringDict [in] Handle to the StringDict object.
 * @param key [in] Unique key of the  item to set.
 * @param item [in]  item to set.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringDict_Set(ID3_DOCUMENT_STRING_DICT hStringDict, const char *key, const char *item);

/**
 * Gets the number of elements in the StringDict object.
 * @param hStringDict [in] Handle to the StringDict object.
 * @param count [out] Number of elements in the  object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringDict_GetCount(ID3_DOCUMENT_STRING_DICT hStringDict, int *count);

/**
 * Determines whether the StringDictobject contains the specified key.
 * @param hStringDict [in] Handle to the StringDict object.
 * @param key [in] The key to locate in the  object.
 * @param result [out] true if the  object contains an element with the specified key, otherwise false.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringDict_ContainsKey(ID3_DOCUMENT_STRING_DICT hStringDict, const char *key, bool *result);

/**
 * Removes an element of the StringDict object.
 * @param hStringDict [in] Handle to the StringDict object.
 * @param key [in] Unique key of the  item to remove.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringDict_Remove(ID3_DOCUMENT_STRING_DICT hStringDict, const char *key);

#ifdef __cplusplus
}
#endif

#endif
