//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentImage_h
#define _id3DocumentImage_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentPixelFormat.h>
#include <id3Document/id3DocumentImage.h>
#include <id3Document/id3DocumentRectangle.h>
#include <id3Document/id3DocumentImageFormat.h>
#include <id3Document/id3DocumentImageBuffer.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a document image.
 */

/**
 * Handle to a DocumentImage object of the Document module.
 */
typedef struct id3DocumentImage *ID3_DOCUMENT_IMAGE;

/**
 * @brief Initializes a DocumentImage object.
 * @param phDocumentImage [in] Handle to the DocumentImage object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_Initialize(ID3_DOCUMENT_IMAGE *phDocumentImage);

/**
 * @brief Releases the memory allocated for a DocumentImage.
 * @param phDocumentImage [in] Handle to the DocumentImage object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_Dispose(ID3_DOCUMENT_IMAGE *phDocumentImage);

/**
 * @brief Checks if the handle object is valid.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_CheckHandle(ID3_DOCUMENT_IMAGE hDocumentImage);

/**
 * @brief Copies an instance of DocumentImage object to another.
 * @param hDocumentImageSrc [in] Handle to the source DocumentImage object.
 * @param hDocumentImageDst [in] Handle to the destination DocumentImage object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_CopyTo(ID3_DOCUMENT_IMAGE hDocumentImageSrc, ID3_DOCUMENT_IMAGE hDocumentImageDst);

/**
 * Gets the raw data buffer of the image.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param data [out] Raw data buffer of the image.
 * @param dataSize [in,out] Size of the 'data' buffer. Receives the number of elements in the 'data' array.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_GetData(ID3_DOCUMENT_IMAGE hDocumentImage, unsigned char *data, int *dataSize);

/**
 * Gets the height in pixels.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param height [out] Height in pixels.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_GetHeight(ID3_DOCUMENT_IMAGE hDocumentImage, int *height);

/**
 * Gets the pixel depth.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param pixelDepth [out] Pixel depth.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_GetPixelDepth(ID3_DOCUMENT_IMAGE hDocumentImage, int *pixelDepth);

/**
 * Gets the pixel format.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param ePixelFormat [out] Pixel format.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_GetPixelFormat(ID3_DOCUMENT_IMAGE hDocumentImage, id3DocumentPixelFormat *ePixelFormat);

/**
 * Gets the stride in bytes.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param stride [out] Stride in bytes.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_GetStride(ID3_DOCUMENT_IMAGE hDocumentImage, int *stride);

/**
 * Gets the width in pixels.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param width [out] Width in pixels.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_GetWidth(ID3_DOCUMENT_IMAGE hDocumentImage, int *width);

/**
 * Computes the difference with another image.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param hCmpDocumentImage [in] The image to compare.
 * @param result [out] Comparison result in percent (0% is same, 100% is full different).
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_Compare(ID3_DOCUMENT_IMAGE hDocumentImage, ID3_DOCUMENT_IMAGE hCmpDocumentImage, float *result);

/**
 * Applies a gamma correction to the image.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param contrast [in] Contrast coefficient, from -255 to 255, to be applied to the image.
 * @param brightness [in] Brightness coefficient, from -255 to 255, to be applid to the image.
 * @param gamma [in] Gamma coefficient, from 0.25 to 2.50, to be applied to the image.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_CorrectGamma(ID3_DOCUMENT_IMAGE hDocumentImage, int contrast, int brightness, float gamma);

/**
 * Crop a rectangular region of interest in the image according to the given bounds.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param left [in] Left bound of the crop to extract.
 * @param top [in] Top bound of the crop to extract.
 * @param width [in] Width of the crop to extract.
 * @param height [in] Height of the crop to extract.
 * @param hDocumentImageCrop [out] The extracted region of interest.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_Crop(ID3_DOCUMENT_IMAGE hDocumentImage, int left, int top, int width, int height, ID3_DOCUMENT_IMAGE hDocumentImageCrop);

/**
 * Downscales the image in-place so that its maximum dimension is equal to the given maximum size, while preserving the aspect ratio.
 * Note: If the maximum dimension is already smaller than the given maximum size, the function does nothing and the scaling ration returned is 1.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param maxSize [in] Maximum image size, in pixels. The value must be greater than 0.
 * @param scaleRatio [out] The scaling ratio applied to the image. Range is ]0:1].
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_Downscale(ID3_DOCUMENT_IMAGE hDocumentImage, int maxSize, float *scaleRatio);

/**
 * Downscales the image so that its maximum dimension equals the given maximum size, while preserving the aspect ratio.
 * Note: If the maximum dimension is already smaller than the given maximum size, the function does nothing and the returned scale ratio is 1.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param hDstDocumentImage [in] Destination image.
 * @param maxSize [in] Maximum image size, in pixels. The value must be greater than 0.
 * @param scaleRatio [out] The scaling ratio applied to the image. Range is ]0:1].
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_DownscaleTo(ID3_DOCUMENT_IMAGE hDocumentImage, ID3_DOCUMENT_IMAGE hDstDocumentImage, int maxSize, float *scaleRatio);

/**
 * Extracts a region of interest in the image according to the given bounds.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param sBounds [in] Bounds of the crop to extract.
 * @param hDocumentImageRoi [out] The extracted region of interest.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_ExtractRoi(ID3_DOCUMENT_IMAGE hDocumentImage, const id3DocumentRectangle *sBounds, ID3_DOCUMENT_IMAGE hDocumentImageRoi);

/**
 * Extracts a region of interest in the image according to the given bounds.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param sBounds [in] Bounds of the crop to extract.
 * @param colorRed [in] Background color red (from 0 to 255)
 * @param colorGreen [in] Background color green (from 0 to 255)
 * @param colorBlue [in] Background color blue (from 0 to 255)
 * @param hDocumentImageRoi [out] The extracted region of interest.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_ExtractRoiWithColor(ID3_DOCUMENT_IMAGE hDocumentImage, const id3DocumentRectangle *sBounds, int colorRed, int colorGreen, int colorBlue, ID3_DOCUMENT_IMAGE hDocumentImageRoi);

/**
 * Flips the image in-place.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param flipHorizontally [in] Value indicating whether the image should be flipped horizontally.
 * @param flipVertically [in] Value indicating whether the image should be flipped vertically.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_Flip(ID3_DOCUMENT_IMAGE hDocumentImage, bool flipHorizontally, bool flipVertically);

/**
 * Flips the image.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param flipHorizontally [in] Value indicating whether the image should be flipped horizontally.
 * @param flipVertically [in] Value indicating whether the image should be flipped vertically.
 * @param hDstDocumentImage [in] The destination Image object that receives the flipped image.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_FlipTo(ID3_DOCUMENT_IMAGE hDocumentImage, bool flipHorizontally, bool flipVertically, ID3_DOCUMENT_IMAGE hDstDocumentImage);

/**
 * Creates an Image from the specified data buffer.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param data [in] A buffer that contains the image data.
 * @param dataSize [in] Size of the 'data' buffer.
 * @param ePixelFormat [in] The destination pixel format to convert the input to.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_FromBuffer(ID3_DOCUMENT_IMAGE hDocumentImage, unsigned char *data, int dataSize, id3DocumentPixelFormat ePixelFormat);

/**
 * Creates an Image from the specified file.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param filepath [in] A string that contains the name of the file from which to create the Image.
 * @param ePixelFormat [in] The pixel format into which to convert the input image.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_FromFile(ID3_DOCUMENT_IMAGE hDocumentImage, const char *filepath, id3DocumentPixelFormat ePixelFormat);

/**
 * Creates an Image from the specified raw data buffer.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param pixels [in] A buffer that contains image pixels.
 * @param pixelsSize [in] Size of the 'pixels' buffer.
 * @param width [in] The width, in pixels, of the image.
 * @param height [in] The height, in pixels, of the image.
 * @param stride [in] The stride, in pixels, of the image.
 * @param eSrcPixelFormat [in] The pixel format of the input image.
 * @param eDstPixelFormat [in] The pixel format into which to convert the input image.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_FromRawBuffer(ID3_DOCUMENT_IMAGE hDocumentImage, unsigned char *pixels, int pixelsSize, int width, int height, int stride, id3DocumentPixelFormat eSrcPixelFormat, id3DocumentPixelFormat eDstPixelFormat);

/**
 * Creates an Image from the specified YUV planes.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param yPlane [in] A buffer that contains the Y plane.
 * @param yPlaneSize [in] Size of the 'yPlane' buffer.
 * @param uPlane [in] A buffer that contains the U plane.
 * @param uPlaneSize [in] Size of the 'uPlane' buffer.
 * @param vPlane [in] A buffer that contains the V plane.
 * @param vPlaneSize [in] Size of the 'vPlane' buffer.
 * @param yWidth [in] The width, in pixels, of the Y plane.
 * @param yHeight [in] The height, in pixels, of the Y plane.
 * @param uvPixelStride [in] The pixel-level stride, in pixels, of the U and V planes.
 * @param uvRowStride [in] The row-level stride, in pixels, of the U and V planes.
 * @param eDstPixelFormat [in] The pixel format into which to convert the input image.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_FromYuvPlanes(ID3_DOCUMENT_IMAGE hDocumentImage, unsigned char *yPlane, int yPlaneSize, unsigned char *uPlane, int uPlaneSize, unsigned char *vPlane, int vPlaneSize, int yWidth, int yHeight, int uvPixelStride, int uvRowStride, id3DocumentPixelFormat eDstPixelFormat);

/**
 * Gets a pointer to the image pixels.
 * Warning: This method is unsafe! The user must handle the pixels pointer carefully.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param pixels [out] Pointer to the image pixels.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_GetPixels(ID3_DOCUMENT_IMAGE hDocumentImage, void **pixels);

/**
 * Add padding around the image
 * Only works for BGR 24 bits, RGB 24 bits and Grayscale 8bits images.
 * For grayscale image, the padding color is the red component.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param top [in] Padding on top.
 * @param left [in] Padding on left.
 * @param bottom [in] Padding on bottom.
 * @param right [in] Padding on right.
 * @param colorRed [in] Padding color red (from 0 to 255)
 * @param colorGreen [in] Padding color green (from 0 to 255)
 * @param colorBlue [in] Padding color blue (from 0 to 255)
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_Pad(ID3_DOCUMENT_IMAGE hDocumentImage, int top, int left, int bottom, int right, int colorRed, int colorGreen, int colorBlue);

/**
 * Reallocates the internal memory of the Image from parameters.
 * Note: If the given parameters are the same as the ones of the object, then there is nothing done in this function.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param width [in] The new width, in pixels.
 * @param height [in] The new height, in pixels.
 * @param ePixelFormat [in] The new pixel format.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_Reallocate(ID3_DOCUMENT_IMAGE hDocumentImage, int width, int height, id3DocumentPixelFormat ePixelFormat);

/**
 * Resizes the image in-place to the required width and height.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param width [in] The new width, in pixels.
 * @param height [in] The new height, in pixels.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_Resize(ID3_DOCUMENT_IMAGE hDocumentImage, int width, int height);

/**
 * Resizes the image to the specified width and height.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param width [in] The new width, in pixels.
 * @param height [in] The new height, in pixels.
 * @param hDstDocumentImage [in] The destination image that receives the resized image.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_ResizeTo(ID3_DOCUMENT_IMAGE hDocumentImage, int width, int height, ID3_DOCUMENT_IMAGE hDstDocumentImage);

/**
 * Rotates the image in-place to the specified angle.
 * Note: The rotation is performed counter-clockwise.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param angle [in] The rotation angle, in degree. Supported values are [0, 90, 180, 270].
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_Rotate(ID3_DOCUMENT_IMAGE hDocumentImage, int angle);

/**
 * Rotates the image to the specified angle.
 * Note: The rotation is performed counter-clockwise.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param angle [in] The rotation angle, in degree. Supported values are [0, 90, 180, 270].
 * @param hDstDocumentImage [in] The destination Image that receives the rotated image.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_RotateTo(ID3_DOCUMENT_IMAGE hDocumentImage, int angle, ID3_DOCUMENT_IMAGE hDstDocumentImage);

/**
 * Sets the Image object fields.
 * Warning: This function does not allocate any memory and is unsafe! The user must handle the pixels pointer carefully.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param width [in] The width, in pixels, of the image.
 * @param height [in] The height, in pixels, of the image.
 * @param ePixelFormat [in] The pixel format of the image.
 * @param pixels [in] A pointer to the image pixels of the image.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_Set(ID3_DOCUMENT_IMAGE hDocumentImage, int width, int height, id3DocumentPixelFormat ePixelFormat, void *pixels);

/**
 * Exports the image to a buffer.
 * The compression level meaning depends on the algorithm used:
 * - For JPEG compression, the value is the expected quality and may vary from 1 to 100.
 * - For JPEG2000 compression, the value is the compression rate and may vary from 1 to 512.
 * - For PNG compression, the value is the compression rate and may vary from 1 to 10.
 * - For all other formats, the value is ignored.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param eDocumentImageFormat [in] The image format to export the image to.
 * @param compressionLevel [in] The compression level to be applied.
 * @param data [out] Buffer that receives the image data.
 * @param dataSize [in,out] Size of the 'data' buffer. Receives the number of elements in the 'data' array.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_ToBuffer(ID3_DOCUMENT_IMAGE hDocumentImage, id3DocumentImageFormat eDocumentImageFormat, float compressionLevel, unsigned char *data, int *dataSize);

/**
 * Copies the image to a buffer in the specified format.
 * The compression level meaning depends on the algorithm used:
 * - For JPEG compression, the value is the expected quality and may vary from 1 to 100.
 * - For JPEG2000 compression, the value is the compression rate and may vary from 1 to 512.
 * - For PNG compression, the value is the compression rate and may vary from 1 to 10.
 * - For all other formats, the value is ignored.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param eDocumentImageFormat [in] The output image format.
 * @param compressionLevel [in] The compression level to be applied.
 * @param hBuffer [in] The image buffer that receives the image data.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_ToBufferObject(ID3_DOCUMENT_IMAGE hDocumentImage, id3DocumentImageFormat eDocumentImageFormat, float compressionLevel, ID3_DOCUMENT_IMAGE_BUFFER hBuffer);

/**
 * Saves the image to the specified file.
 * The compression level meaning depends on the algorithm used:
 * - For JPEG compression, the value is the expected quality and may vary from 1 to 100.
 * - For JPEG2000 compression, the value is the compression rate and may vary from 1 to 512.
 * - For PNG compression, the value is the compression rate and may vary from 1 to 10.
 * - For all other formats, the value is ignored.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param filepath [in] A string that contains the name of the file to which to save the image.
 * @param compressionLevel [in] The compression level to be applied.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_ToFile(ID3_DOCUMENT_IMAGE hDocumentImage, const char *filepath, float compressionLevel);

/**
 * Copies image pixels in the specified format into a buffer.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @param ePixelFormat [in] The pixel format.
 * @param hBuffer [in] Buffer that receives the raw image data.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_ToRawBuffer(ID3_DOCUMENT_IMAGE hDocumentImage, id3DocumentPixelFormat ePixelFormat, ID3_DOCUMENT_IMAGE_BUFFER hBuffer);

/**
 * Transposes the image object.
 * @param hDocumentImage [in] Handle to the DocumentImage object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImage_Transpose(ID3_DOCUMENT_IMAGE hDocumentImage);

#ifdef __cplusplus
}
#endif

#endif
