//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentImageList_h
#define _id3DocumentImageList_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentImage.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Handle to a ImageList object of the Document module.
 */
typedef struct id3DocumentImageList *ID3_DOCUMENT_IMAGE_LIST;

/**
 * @brief Initializes a ImageList object.
 * @param phImageList [in] Handle to the ImageList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImageList_Initialize(ID3_DOCUMENT_IMAGE_LIST *phImageList);

/**
 * @brief Releases the memory allocated for a ImageList.
 * @param phImageList [in] Handle to the ImageList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImageList_Dispose(ID3_DOCUMENT_IMAGE_LIST *phImageList);

/**
 * @brief Checks if the handle object is valid.
 * @param hImageList [in] Handle to the ImageList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImageList_CheckHandle(ID3_DOCUMENT_IMAGE_LIST hImageList);

/**
 * @brief Copies an instance of ImageList object to another.
 * @param hImageListSrc [in] Handle to the source ImageList object.
 * @param hImageListDst [in] Handle to the destination ImageList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImageList_CopyTo(ID3_DOCUMENT_IMAGE_LIST hImageListSrc, ID3_DOCUMENT_IMAGE_LIST hImageListDst);

/**
 * Clears the ImageList object.
 * @param hImageList [in] Handle to the ImageList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImageList_Clear(ID3_DOCUMENT_IMAGE_LIST hImageList);

/**
 * Adds an item to the ImageList object.
 * @param hImageList [in] Handle to the ImageList object.
 * @param hImageItem [in] Image item to add.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImageList_Add(ID3_DOCUMENT_IMAGE_LIST hImageList, ID3_DOCUMENT_IMAGE hImageItem);

/**
 * Gets an item of the ImageList object.
 * @param hImageList [in] Handle to the ImageList object.
 * @param index [in] Index of the Image item to get.
 * @param hImageItem [out] Image item to get.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImageList_Get(ID3_DOCUMENT_IMAGE_LIST hImageList, int index, ID3_DOCUMENT_IMAGE hImageItem);

/**
 * Sets an item of the ImageList object.
 * @param hImageList [in] Handle to the ImageList object.
 * @param index [in] Index of the Image item to set.
 * @param hImageItem [in] Image item to set.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImageList_Set(ID3_DOCUMENT_IMAGE_LIST hImageList, int index, ID3_DOCUMENT_IMAGE hImageItem);

/**
 * Gets the number of elements in the ImageList object.
 * @param hImageList [in] Handle to the ImageList object.
 * @param count [out] Number of elements in the Image object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImageList_GetCount(ID3_DOCUMENT_IMAGE_LIST hImageList, int *count);

/**
 * Removes an element of the ImageList object.
 * @param hImageList [in] Handle to the ImageList object.
 * @param index [in] Index of the Image item to remove.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImageList_RemoveAt(ID3_DOCUMENT_IMAGE_LIST hImageList, int index);

/**
 * ImageList object.
 * @param hImageList [in] Handle to the ImageList object.
 * @param newSize [in] Image object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentImageList_Resize(ID3_DOCUMENT_IMAGE_LIST hImageList, int newSize);

#ifdef __cplusplus
}
#endif

#endif
