//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentMrzReadingResult_h
#define _id3DocumentMrzReadingResult_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentPointList.h>
#include <id3Document/id3DocumentMrzType.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents the result of MRZ reading.
 */

/**
 * Handle to a MrzReadingResult object of the Document module.
 */
typedef struct id3DocumentMrzReadingResult *ID3_DOCUMENT_MRZ_READING_RESULT;

/**
 * @brief Initializes a MrzReadingResult object.
 * @param phMrzReadingResult [in] Handle to the MrzReadingResult object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReadingResult_Initialize(ID3_DOCUMENT_MRZ_READING_RESULT *phMrzReadingResult);

/**
 * @brief Releases the memory allocated for a MrzReadingResult.
 * @param phMrzReadingResult [in] Handle to the MrzReadingResult object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReadingResult_Dispose(ID3_DOCUMENT_MRZ_READING_RESULT *phMrzReadingResult);

/**
 * @brief Checks if the handle object is valid.
 * @param hMrzReadingResult [in] Handle to the MrzReadingResult object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReadingResult_CheckHandle(ID3_DOCUMENT_MRZ_READING_RESULT hMrzReadingResult);

/**
 * @brief Copies an instance of MrzReadingResult object to another.
 * @param hMrzReadingResultSrc [in] Handle to the source MrzReadingResult object.
 * @param hMrzReadingResultDst [in] Handle to the destination MrzReadingResult object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReadingResult_CopyTo(ID3_DOCUMENT_MRZ_READING_RESULT hMrzReadingResultSrc, ID3_DOCUMENT_MRZ_READING_RESULT hMrzReadingResultDst);

/**
 * Gets the boundaries of the MRZ detected, in following order: top-left, top-right, bottom-right, bottom-left.
 * @param hMrzReadingResult [in] Handle to the MrzReadingResult object.
 * @param hBounds [out] Boundaries of the MRZ detected, in following order: top-left, top-right, bottom-right, bottom-left.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReadingResult_GetBounds(ID3_DOCUMENT_MRZ_READING_RESULT hMrzReadingResult, ID3_DOCUMENT_POINT_LIST hBounds);

/**
 * Gets the confidence score for the MRZ detected.
 * @param hMrzReadingResult [in] Handle to the MrzReadingResult object.
 * @param confidence [out] Confidence score for the MRZ detected.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReadingResult_GetConfidence(ID3_DOCUMENT_MRZ_READING_RESULT hMrzReadingResult, float *confidence);

/**
 * Gets the value of the MRZ detected.
 * @param hMrzReadingResult [in] Handle to the MrzReadingResult object.
 * @param mrz [out] Value of the MRZ detected.
 * @param mrzSize [in,out] Size of the 'mrz' buffer. Receives the minimum required buffer size to hold the 'mrz' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'mrz' (excluding terminating NULL character).
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReadingResult_GetMrz(ID3_DOCUMENT_MRZ_READING_RESULT hMrzReadingResult, char *mrz, int *mrzSize);

/**
 * Gets the type of the MRZ detected.
 * @param hMrzReadingResult [in] Handle to the MrzReadingResult object.
 * @param eMrzType [out] Type of the MRZ detected.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentMrzReadingResult_GetMrzType(ID3_DOCUMENT_MRZ_READING_RESULT hMrzReadingResult, id3DocumentMrzType *eMrzType);

#ifdef __cplusplus
}
#endif

#endif
