//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentStringList_h
#define _id3DocumentStringList_h

#include <id3Document/id3DocumentExports.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Represents a string lists.
 */

/**
 * Handle to a StringList object of the Document module.
 */
typedef struct id3DocumentStringList *ID3_DOCUMENT_STRING_LIST;

/**
 * @brief Initializes a StringList object.
 * @param phStringList [in] Handle to the StringList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringList_Initialize(ID3_DOCUMENT_STRING_LIST *phStringList);

/**
 * @brief Releases the memory allocated for a StringList.
 * @param phStringList [in] Handle to the StringList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringList_Dispose(ID3_DOCUMENT_STRING_LIST *phStringList);

/**
 * @brief Checks if the handle object is valid.
 * @param hStringList [in] Handle to the StringList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringList_CheckHandle(ID3_DOCUMENT_STRING_LIST hStringList);

/**
 * @brief Copies an instance of StringList object to another.
 * @param hStringListSrc [in] Handle to the source StringList object.
 * @param hStringListDst [in] Handle to the destination StringList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringList_CopyTo(ID3_DOCUMENT_STRING_LIST hStringListSrc, ID3_DOCUMENT_STRING_LIST hStringListDst);

/**
 * Clears the StringList object.
 * @param hStringList [in] Handle to the StringList object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringList_Clear(ID3_DOCUMENT_STRING_LIST hStringList);

/**
 * Adds an item to the StringList object.
 * @param hStringList [in] Handle to the StringList object.
 * @param item [in]  item to add.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringList_Add(ID3_DOCUMENT_STRING_LIST hStringList, const char *item);

/**
 * Gets an item of the StringList object.
 * @param hStringList [in] Handle to the StringList object.
 * @param index [in] Index of the  item to get.
 * @param item [out]  item to get.
 * @param itemSize [in,out] Size of the 'item' buffer. Receives the minimum required buffer size to hold the 'item' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'item' (excluding terminating NULL character).
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringList_Get(ID3_DOCUMENT_STRING_LIST hStringList, int index, char *item, int *itemSize);

/**
 * Sets an item of the StringList object.
 * @param hStringList [in] Handle to the StringList object.
 * @param index [in] Index of the  item to set.
 * @param item [in]  item to set.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringList_Set(ID3_DOCUMENT_STRING_LIST hStringList, int index, const char *item);

/**
 * Gets the number of elements in the StringList object.
 * @param hStringList [in] Handle to the StringList object.
 * @param count [out] Number of elements in the  object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringList_GetCount(ID3_DOCUMENT_STRING_LIST hStringList, int *count);

/**
 * Removes an element of the StringList object.
 * @param hStringList [in] Handle to the StringList object.
 * @param index [in] Index of the  item to remove.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringList_RemoveAt(ID3_DOCUMENT_STRING_LIST hStringList, int index);

/**
 * StringList object.
 * @param hStringList [in] Handle to the StringList object.
 * @param newSize [in]  object.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentStringList_Resize(ID3_DOCUMENT_STRING_LIST hStringList, int newSize);

#ifdef __cplusplus
}
#endif

#endif
