//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------

/// Enumerates possible error codes.
enum DocumentError {
	/// No error.
	success,
	/// The given buffer is too small.
	insufficientBuffer,
	/// The requested object does not exist; a 'null' should be returned if possible.
	objectDoesNotExist,
	/// Base error code of the Document module.
	base,
	/// Minimum error code of the Document module.
	minimum,
	/// The handle is invalid.
	invalidHandle,
	/// The parameter is invalid.
	invalidParameter,
	/// The image pixel format is invalid.
	invalidPixelFormat,
	/// The model could not be found in the given directory path.
	invalidModelPath,
	/// The document template could not be found at the given path.
	invalidDocumentTemplatePath,
	/// The model which is required for this operation has not been  loaded.
	unloadedModel,
	/// The document template required for this operation has not been loaded.
	unloadedDocumentTemplate,
	/// The required element is not present in the document annotation.
	unknownAnnotation,
	/// The format of the document template is invalid.
	invalidDocumentTemplateFormat,
	/// The frame buffer is not full, and computation is not allowed.
	insufficientFrameNumber,
	/// One of our computation backends failed for unknown reasons.
	backendError,
	/// The inference backend failed to infer, probably due to inconsistent input.
	inferenceBackendError,
	/// The requested processing unit is not available in this build.
	processingUnitUnavailable,
	/// This function is not implemented.
	functionNotImplemented,
}

extension DocumentErrorX on DocumentError {
	/// Creates an enumeration value from int.
	/// param value The int value.
	/// return The created DocumentError.
  	static DocumentError fromValue(int value) {
    	switch (value) {
      	case 0:
        	return DocumentError.success;
      	case -1:
        	return DocumentError.insufficientBuffer;
      	case -2:
        	return DocumentError.objectDoesNotExist;
      	case -11000:
        	return DocumentError.base;
      	case -11999:
        	return DocumentError.minimum;
      	case -11001:
        	return DocumentError.invalidHandle;
      	case -11002:
        	return DocumentError.invalidParameter;
      	case -11004:
        	return DocumentError.invalidPixelFormat;
      	case -11008:
        	return DocumentError.invalidModelPath;
      	case -11009:
        	return DocumentError.invalidDocumentTemplatePath;
      	case -11020:
        	return DocumentError.unloadedModel;
      	case -11021:
        	return DocumentError.unloadedDocumentTemplate;
      	case -11022:
        	return DocumentError.unknownAnnotation;
      	case -11023:
        	return DocumentError.invalidDocumentTemplateFormat;
      	case -11030:
        	return DocumentError.insufficientFrameNumber;
      	case -11039:
        	return DocumentError.backendError;
      	case -11040:
        	return DocumentError.inferenceBackendError;
      	case -11041:
        	return DocumentError.processingUnitUnavailable;
      	case -11056:
        	return DocumentError.functionNotImplemented;
	    default:
        	throw Exception("Unable to find the specified enum value for $value");
    	}
  	}

	/// Gets the enumeration value as int.
	/// return The value.
  	int get value {
    	switch (this) {
      	case DocumentError.success:
        	return 0;
      	case DocumentError.insufficientBuffer:
        	return -1;
      	case DocumentError.objectDoesNotExist:
        	return -2;
      	case DocumentError.base:
        	return -11000;
      	case DocumentError.minimum:
        	return -11999;
      	case DocumentError.invalidHandle:
        	return -11001;
      	case DocumentError.invalidParameter:
        	return -11002;
      	case DocumentError.invalidPixelFormat:
        	return -11004;
      	case DocumentError.invalidModelPath:
        	return -11008;
      	case DocumentError.invalidDocumentTemplatePath:
        	return -11009;
      	case DocumentError.unloadedModel:
        	return -11020;
      	case DocumentError.unloadedDocumentTemplate:
        	return -11021;
      	case DocumentError.unknownAnnotation:
        	return -11022;
      	case DocumentError.invalidDocumentTemplateFormat:
        	return -11023;
      	case DocumentError.insufficientFrameNumber:
        	return -11030;
      	case DocumentError.backendError:
        	return -11039;
      	case DocumentError.inferenceBackendError:
        	return -11040;
      	case DocumentError.processingUnitUnavailable:
        	return -11041;
      	case DocumentError.functionNotImplemented:
        	return -11056;
		}
  	}
}
