//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
// ignore_for_file: unused_import
import 'dart:collection';
import 'dart:ffi';
import 'dart:typed_data';
import 'package:ffi/ffi.dart';
import 'document_sdk_generated_bindings.dart';
import 'document_native.dart';

import '../id3_document.dart';

final _finalizer = NativeFinalizer(documentSDK.addresses.id3DocumentInfoDictionary_Dispose.cast());

/// Represents a dictionary of document information.
class DocumentInfoDictionary extends Object with IterableMixin<DocumentInfo> implements Finalizable {
  /// Native handle.
  late Pointer<Pointer<id3DocumentInfoDictionary>> _pHandle;
  bool _disposable = true;

  /// Gets the native handle.
  /// return The native handle.
  Pointer<id3DocumentInfoDictionary> get handle => _pHandle.value;

  /// Creates a new instance of the DocumentInfoDictionary class.
  ///
  /// throws DocumentException An error has occurred during Document Library execution.
  DocumentInfoDictionary() {
    _pHandle = calloc();
    try {
      var err = documentSDK.id3DocumentInfoDictionary_Initialize(_pHandle);
      if (err != DocumentError.success.value) {
        throw DocumentException(err);
      }
      _finalizer.attach(this, _pHandle.cast(), detach: this);
    } finally {}
  }

  /// Creates a new instance of the DocumentInfoDictionary class.
  ///
  /// param handle     Handle to the DocumentInfoDictionary.
  /// throws DocumentException An error has occurred during Document Library execution.
  DocumentInfoDictionary.fromHandle(Pointer<id3DocumentInfoDictionary> handle) {
    _pHandle = calloc();
    _pHandle.value = handle;
    _disposable = false;
  }

  /// Releases all resources used by this DocumentInfoDictionary.
  void dispose() {
    if (_disposable) {
      documentSDK.id3DocumentInfoDictionary_Dispose(_pHandle);
      calloc.free(_pHandle);
      _finalizer.detach(this);
    }
  }

  // Copyable methods

  /// Creates a copy of the DocumentInfoDictionary object.
  ///
  /// return The newly created DocumentInfoDictionary object.
  /// throws DocumentException An error has occurred during Document Library execution.
  DocumentInfoDictionary clone() {
    DocumentInfoDictionary clone = DocumentInfoDictionary();
    var err = documentSDK.id3DocumentInfoDictionary_CopyTo(_pHandle.value, clone.handle);
    if (err != DocumentError.success.value) {
      throw DocumentException(err);
    }
    return clone;
  }

  /// Gets or sets the element at the specified index.
  operator [](String key) => get(key);
  operator []=(String key, DocumentInfo item) => set(key, item);

  /// Gets the number of elements contained in the list or dictionary.
  int get count => getCount();

  @override
  Iterator<DocumentInfo> get iterator => DocumentInfoDictionaryIterator(this);

  ///
  /// Gets a string list containing the keys in the dict.
  ///
  /// throws DocumentException An error has occurred during Document Library execution.
  StringList get keys => getKeys();

  // Public methods
  /// Clears the DocumentInfoDictionary object.
  ///
  /// throws DocumentException An error has occurred during Document Library execution.
  void clear() {
    var err = documentSDK.id3DocumentInfoDictionary_Clear(_pHandle.value);
    if (err != DocumentError.success.value) {
      throw DocumentException(err);
    }
  }

  /// Adds an item to the DocumentInfoDictionary object.
  ///
  /// param key Unique key of the DocumentInfo item to add.
  /// param documentInfoItem DocumentInfo item to add.
  /// throws DocumentException An error has occurred during Document Library execution.
  void add(String? key, DocumentInfo documentInfoItem) {
    Pointer<Char>? pKey = key?.toNativeUtf8().cast<Char>();
    try {
      var err = documentSDK.id3DocumentInfoDictionary_Add(_pHandle.value, pKey ?? nullptr, documentInfoItem.handle);
      if (err != DocumentError.success.value) {
        throw DocumentException(err);
      }
    } finally {
      if (pKey != null) {
        calloc.free(pKey);
      }
    }
  }

  /// Gets an item of the DocumentInfoDictionary object.
  ///
  /// param key Unique key of the DocumentInfo item to get.
  /// return DocumentInfo item to get.
  /// throws DocumentException An error has occurred during Document Library execution.
  DocumentInfo get(String? key) {
    Pointer<Char>? pKey = key?.toNativeUtf8().cast<Char>();
    DocumentInfo documentInfoItem = DocumentInfo();
    try {
      var err = documentSDK.id3DocumentInfoDictionary_Get(_pHandle.value, pKey ?? nullptr, documentInfoItem.handle);
      if (err != DocumentError.success.value) {
        documentInfoItem.dispose();
        throw DocumentException(err);
      }
      return documentInfoItem;
    } finally {
      if (pKey != null) {
        calloc.free(pKey);
      }
    }
  }

  /// Sets an item of the DocumentInfoDictionary object.
  ///
  /// param key Unique key of the DocumentInfo item to set.
  /// param documentInfoItem DocumentInfo item to set.
  /// throws DocumentException An error has occurred during Document Library execution.
  void set(String? key, DocumentInfo documentInfoItem) {
    Pointer<Char>? pKey = key?.toNativeUtf8().cast<Char>();
    try {
      var err = documentSDK.id3DocumentInfoDictionary_Set(_pHandle.value, pKey ?? nullptr, documentInfoItem.handle);
      if (err != DocumentError.success.value) {
        throw DocumentException(err);
      }
    } finally {
      if (pKey != null) {
        calloc.free(pKey);
      }
    }
  }

  /// Gets the number of elements in the DocumentInfoDictionary object.
  ///
  /// return Number of elements in the DocumentInfo object.
  /// throws DocumentException An error has occurred during Document Library execution.
  int getCount() {
    Pointer<Int> pCount = calloc();
    try {
      var err = documentSDK.id3DocumentInfoDictionary_GetCount(_pHandle.value, pCount);
      if (err != DocumentError.success.value) {
        throw DocumentException(err);
      }
      final vCount = pCount.value;
      return vCount;
    } finally {
      calloc.free(pCount);
    }
  }

  /// Determines whether the DocumentInfoDictionaryobject contains the specified key.
  ///
  /// param key The key to locate in the DocumentInfo object.
  /// return true if the DocumentInfo object contains an element with the specified key, otherwise false.
  /// throws DocumentException An error has occurred during Document Library execution.
  bool containsKey(String? key) {
    Pointer<Char>? pKey = key?.toNativeUtf8().cast<Char>();
    Pointer<Bool> pResult = calloc();
    try {
      var err = documentSDK.id3DocumentInfoDictionary_ContainsKey(_pHandle.value, pKey ?? nullptr, pResult);
      if (err != DocumentError.success.value) {
        throw DocumentException(err);
      }
      final vResult = pResult.value;
      return vResult;
    } finally {
      if (pKey != null) {
        calloc.free(pKey);
      }
      calloc.free(pResult);
    }
  }

  /// Removes an element of the DocumentInfoDictionary object.
  ///
  /// param key Unique key of the DocumentInfo item to remove.
  /// throws DocumentException An error has occurred during Document Library execution.
  void remove(String? key) {
    Pointer<Char>? pKey = key?.toNativeUtf8().cast<Char>();
    try {
      var err = documentSDK.id3DocumentInfoDictionary_Remove(_pHandle.value, pKey ?? nullptr);
      if (err != DocumentError.success.value) {
        throw DocumentException(err);
      }
    } finally {
      if (pKey != null) {
        calloc.free(pKey);
      }
    }
  }

  /// Gets the gets a string list containing the keys in the dict.
  ///
  /// return Gets a string list containing the keys in the dict.
  /// throws DocumentException An error has occurred during Document Library execution.
  StringList getKeys() {
    StringList keys = StringList();
    var err = documentSDK.id3DocumentInfoDictionary_GetKeys(_pHandle.value, keys.handle);
    if (err != DocumentError.success.value) {
      keys.dispose();
      throw DocumentException(err);
    }
    return keys;
  }

}

class DocumentInfoDictionaryIterator implements Iterator<DocumentInfo> {
  DocumentInfoDictionaryIterator(this._list)
      : _count = _list.getCount(),
        _keys = [] {
    final StringList keys = _list.getKeys();
    for (var key in keys) {
      _keys.add(key);
    }
    keys.dispose();
  }
  final DocumentInfoDictionary _list;
  final int _count;
  int _index = -1;
  final List<String> _keys;

  @override
  DocumentInfo get current => _list.get(_keys[_index]);

  @override
  bool moveNext() {
    _index++;
    return _index < _count;
  }
}
