//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------

/// Enumerates possible error codes related to image management.
enum ImageError {
	/// No error.
	success,
	/// The given buffer is too small.
	insufficientBuffer,
	/// The requested object does not exist; a 'null' should be returned if possible.
	objectDoesNotExist,
	/// Base error code for Image module.
	base,
	/// Minimum error code for Image module.
	minimum,
	/// The handle is invalid.
	invalidHandle,
	/// The parameter is invalid.
	invalidParameter,
	/// The image sizes are not compatible for the requested operation.
	imageSizeMismatch,
	/// The pixel formats are not compatible.
	pixelFormatMismatch,
	/// The given key is already used in the dictionary. Multiple values per key is not supported.
	keyAlreadyExists,
	/// The given key is not present in the dictionary.
	keyNotFound,
	/// The required pixel format is not correct.
	invalidPixelFormat,
	/// An error as occured during python module initialization.
	pythonModule,
	/// Function not implemented.
	functionNotImplemented,
	/// TODO
	backendError,
}

extension ImageErrorX on ImageError {
	/// Creates an enumeration value from int.
	/// param value The int value.
	/// return The created ImageError.
  	static ImageError fromValue(int value) {
    	switch (value) {
      	case 0:
        	return ImageError.success;
      	case -1:
        	return ImageError.insufficientBuffer;
      	case -2:
        	return ImageError.objectDoesNotExist;
      	case -3000:
        	return ImageError.base;
      	case -3999:
        	return ImageError.minimum;
      	case -3001:
        	return ImageError.invalidHandle;
      	case -3002:
        	return ImageError.invalidParameter;
      	case -3003:
        	return ImageError.imageSizeMismatch;
      	case -3004:
        	return ImageError.pixelFormatMismatch;
      	case -3005:
        	return ImageError.keyAlreadyExists;
      	case -3006:
        	return ImageError.keyNotFound;
      	case -3007:
        	return ImageError.invalidPixelFormat;
      	case -3008:
        	return ImageError.pythonModule;
      	case -3009:
        	return ImageError.functionNotImplemented;
      	case -3010:
        	return ImageError.backendError;
	    default:
        	throw Exception("Unable to find the specified enum value for $value");
    	}
  	}

	/// Gets the enumeration value as int.
	/// return The value.
  	int get value {
    	switch (this) {
      	case ImageError.success:
        	return 0;
      	case ImageError.insufficientBuffer:
        	return -1;
      	case ImageError.objectDoesNotExist:
        	return -2;
      	case ImageError.base:
        	return -3000;
      	case ImageError.minimum:
        	return -3999;
      	case ImageError.invalidHandle:
        	return -3001;
      	case ImageError.invalidParameter:
        	return -3002;
      	case ImageError.imageSizeMismatch:
        	return -3003;
      	case ImageError.pixelFormatMismatch:
        	return -3004;
      	case ImageError.keyAlreadyExists:
        	return -3005;
      	case ImageError.keyNotFound:
        	return -3006;
      	case ImageError.invalidPixelFormat:
        	return -3007;
      	case ImageError.pythonModule:
        	return -3008;
      	case ImageError.functionNotImplemented:
        	return -3009;
      	case ImageError.backendError:
        	return -3010;
		}
  	}
}
