//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
// ignore_for_file: unused_import
import 'dart:convert';
import 'dart:ffi';
import 'dart:typed_data';
import 'package:ffi/ffi.dart';
import 'document_sdk_generated_bindings.dart';
import 'document_native.dart';

import '../id3_document.dart';

final _finalizer = NativeFinalizer(documentSDK.addresses.id3DocumentMrzReadingResult_Dispose.cast());

/// Represents the result of MRZ reading.
class MrzReadingResult implements Finalizable {
  /// Native handle.
  late Pointer<Pointer<id3DocumentMrzReadingResult>> _pHandle;
  bool _disposable = true;

  /// Gets the native handle.
  /// return The native handle.
  Pointer<id3DocumentMrzReadingResult> get handle => _pHandle.value;

  /// Creates a new instance of the MrzReadingResult class.
  ///
  /// throws DocumentException An error has occurred during Document Library execution.
  MrzReadingResult() {
    _pHandle = calloc();
    try {
      var err = documentSDK.id3DocumentMrzReadingResult_Initialize(_pHandle);
      if (err != DocumentError.success.value) {
        throw DocumentException(err);
      }
      _finalizer.attach(this, _pHandle.cast(), detach: this);
    } finally {}
  }

  /// Creates a new instance of the MrzReadingResult class.
  ///
  /// param handle     Handle to the MrzReadingResult.
  /// throws DocumentException An error has occurred during Document Library execution.
  MrzReadingResult.fromHandle(Pointer<id3DocumentMrzReadingResult> handle) {
    _pHandle = calloc();
    _pHandle.value = handle;
    _disposable = false;
  }

  /// Releases all resources used by this MrzReadingResult.
  void dispose() {
    if (_disposable) {
      documentSDK.id3DocumentMrzReadingResult_Dispose(_pHandle);
      calloc.free(_pHandle);
      _finalizer.detach(this);
    }
  }

  // Copyable methods

  /// Creates a copy of the MrzReadingResult object.
  ///
  /// return The newly created MrzReadingResult object.
  /// throws DocumentException An error has occurred during Document Library execution.
  MrzReadingResult clone() {
    MrzReadingResult clone = MrzReadingResult();
    var err = documentSDK.id3DocumentMrzReadingResult_CopyTo(_pHandle.value, clone.handle);
    if (err != DocumentError.success.value) {
      throw DocumentException(err);
    }
    return clone;
  }


  ///
  /// Boundaries of the MRZ detected, in following order: top-left, top-right, bottom-right, bottom-left.
  ///
  /// throws DocumentException An error has occurred during Document Library execution.
  PointList get bounds => getBounds();

  ///
  /// Confidence score for the MRZ detected.
  ///
  /// throws DocumentException An error has occurred during Document Library execution.
  double get confidence => getConfidence();

  ///
  /// Value of the MRZ detected.
  ///
  /// throws DocumentException An error has occurred during Document Library execution.
  String get mrz => getMrz();

  ///
  /// Type of the MRZ detected.
  ///
  /// throws DocumentException An error has occurred during Document Library execution.
  MrzType get mrzType => getMrzType();

  // Public methods
  /// Gets the boundaries of the MRZ detected, in following order: top-left, top-right, bottom-right, bottom-left.
  ///
  /// return Boundaries of the MRZ detected, in following order: top-left, top-right, bottom-right, bottom-left.
  /// throws DocumentException An error has occurred during Document Library execution.
  PointList getBounds() {
    PointList bounds = PointList();
    var err = documentSDK.id3DocumentMrzReadingResult_GetBounds(_pHandle.value, bounds.handle);
    if (err != DocumentError.success.value) {
      bounds.dispose();
      throw DocumentException(err);
    }
    return bounds;
  }

  /// Gets the confidence score for the MRZ detected.
  ///
  /// return Confidence score for the MRZ detected.
  /// throws DocumentException An error has occurred during Document Library execution.
  double getConfidence() {
    Pointer<Float> pConfidence = calloc();
    try {
      var err = documentSDK.id3DocumentMrzReadingResult_GetConfidence(_pHandle.value, pConfidence);
      if (err != DocumentError.success.value) {
        throw DocumentException(err);
      }
      final vConfidence = pConfidence.value;
      return vConfidence;
    } finally {
      calloc.free(pConfidence);
    }
  }

  /// Gets the value of the MRZ detected.
  ///
  /// return Value of the MRZ detected.
  /// throws DocumentException An error has occurred during Document Library execution.
  String getMrz() {
    Pointer<Char> pMrz = nullptr;
    Pointer<Int> pMrzSize = calloc.allocate(1);
    pMrzSize[0] = -1;
    try {
      var err = documentSDK.id3DocumentMrzReadingResult_GetMrz(_pHandle.value, pMrz, pMrzSize);
      if (err == DocumentError.insufficientBuffer.value) {
        pMrz = calloc.allocate(pMrzSize.value);
        err = documentSDK.id3DocumentMrzReadingResult_GetMrz(_pHandle.value, pMrz, pMrzSize);
        if (err != DocumentError.success.value) {
          throw DocumentException(err);
        }
      }
      final vMrz = utf8.decode(Uint8List.fromList(pMrz.cast<Uint8>().asTypedList(pMrzSize.value)));
      return vMrz;
    } finally {
      calloc.free(pMrz);
      calloc.free(pMrzSize);
    }
  }

  /// Gets the type of the MRZ detected.
  ///
  /// return Type of the MRZ detected.
  /// throws DocumentException An error has occurred during Document Library execution.
  MrzType getMrzType() {
    Pointer<Int32> pMrzType = calloc();
    try {
      var err = documentSDK.id3DocumentMrzReadingResult_GetMrzType(_pHandle.value, pMrzType);
      if (err != DocumentError.success.value) {
        throw DocumentException(err);
      }
      final vMrzType = MrzTypeX.fromValue(pMrzType.value);
      return vMrzType;
    } finally {
      calloc.free(pMrzType);
    }
  }

}

