//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------

/// Enumerates supported MRZ types.
enum MrzType {
	/// No MRZ. Default value.
	none,
	/// Document TD1 (ICAO 9303-5) - 3x30 lines MRZ.
	td1,
	/// Document TD2 (ICAO 9303-6) - 2x36 lines MRZ.
	td2,
	/// Document TD3 (Passports)(ICAO 9303-4) - 2x44 lines MRZ.
	td3,
	/// Driving licence (ISO-18013) - 1x30 line MRZ.
	drivingLicense,
	/// Visa format A (ICAO 9303-7) - 2x44 lines MRZ.
	mrvA,
	/// Visa format B (ICAO 9303-7) - 2x36 lines MRZ.
	mrvB,
	/// Other MRZ Type, with a standard layout but unknown content.
	other,
	/// Old french ID card (PRADO FRA-BO-02002). 2x36 lines MRZ.
	frenchId,
}

extension MrzTypeX on MrzType {
	/// Creates an enumeration value from int.
	/// param value The int value.
	/// return The created MrzType.
  	static MrzType fromValue(int value) {
    	switch (value) {
      	case 0:
        	return MrzType.none;
      	case 1:
        	return MrzType.td1;
      	case 2:
        	return MrzType.td2;
      	case 3:
        	return MrzType.td3;
      	case 4:
        	return MrzType.drivingLicense;
      	case 5:
        	return MrzType.mrvA;
      	case 6:
        	return MrzType.mrvB;
      	case 100:
        	return MrzType.other;
      	case 101:
        	return MrzType.frenchId;
	    default:
        	throw Exception("Unable to find the specified enum value for $value");
    	}
  	}

	/// Gets the enumeration value as int.
	/// return The value.
  	int get value {
    	switch (this) {
      	case MrzType.none:
        	return 0;
      	case MrzType.td1:
        	return 1;
      	case MrzType.td2:
        	return 2;
      	case MrzType.td3:
        	return 3;
      	case MrzType.drivingLicense:
        	return 4;
      	case MrzType.mrvA:
        	return 5;
      	case MrzType.mrvB:
        	return 6;
      	case MrzType.other:
        	return 100;
      	case MrzType.frenchId:
        	return 101;
		}
  	}
}
