//--------------------------------------------------------------------------------------------------
// Copyright (c) id3 Technologies
// All Rights Reserved.
//--------------------------------------------------------------------------------------------------
#ifndef _id3DocumentLibrary_h
#define _id3DocumentLibrary_h

#include <id3Document/id3DocumentExports.h>
#include <id3Document/id3DocumentLibVersion.h>
#include <id3Document/id3DocumentModel.h>
#include <id3Document/id3DocumentProcessingUnit.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * Initializes the SDK library.
 */

/**
 * Retrieves the library version as a 'X.Y.Z' formatted string.
 * @param libraryVersion [out] A string that identifies the library version.
 * @param libraryVersionSize [in,out] Size of the 'libraryVersion' buffer. Receives the minimum required buffer size to hold the 'libraryVersion' string if the given one is too small (including the terminating NULL character), otherwise receives the number of characters in 'libraryVersion' (excluding terminating NULL character).
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentLibrary_GetVersion(char *libraryVersion, int *libraryVersionSize);

/**
 * Retrieves the library version as a structure.
 * @param sVersion [out] The version.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentLibrary_GetVersionEx(id3DocumentLibVersion *sVersion);

/**
 * Loads the specified document template from a file.
 * @param path [in] A string that contains the full path to the document template.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentLibrary_LoadDocumentTemplate(const char *path);

/**
 * Loads a document template from a data buffer.
 * @param buffer [in] A buffer containing the document template data.
 * @param bufferSize [in] Size of the 'buffer' buffer.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentLibrary_LoadDocumentTemplateBuffer(unsigned char *buffer, int bufferSize);

/**
 * Loads a model into memory from the specified directory.
 * @param modelPath [in] The path to directory containing the AI model files.
 * @param eDocumentModel [in] The AI model to be loaded.
 * @param eProcessingUnit [in] Processing unit to load the model into.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentLibrary_LoadModel(const char *modelPath, id3DocumentModel eDocumentModel, id3DocumentProcessingUnit eProcessingUnit);

/**
 * Loads a model into memory from the specified buffer.
 * @param modelBuffer [in] A buffer containing the AI model to be loaded.
 * @param modelBufferSize [in] Size of the 'modelBuffer' buffer.
 * @param eDocumentModel [in] Doc model to load.
 * @param eProcessingUnit [in] The processing unit to use for the model.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentLibrary_LoadModelBuffer(unsigned char *modelBuffer, int modelBufferSize, id3DocumentModel eDocumentModel, id3DocumentProcessingUnit eProcessingUnit);

/**
 * Unloads a document template.
 * @param documentName [in] The document name.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentLibrary_UnloadDocumentTemplate(const char *documentName);

/**
 * Unloads a model from memory.
 * @param eDocumentModel [in] The AI model to be unloaded.
 * @param eProcessingUnit [in] The processing unit used for the model.
 * @return An error code. See id3DocumentError file for the list of possible error codes.
 * @retval id3DocumentError_Success The function succeeded.
 */
ID3DOCUMENT_C_API int ID3_CDECL id3DocumentLibrary_UnloadModel(id3DocumentModel eDocumentModel, id3DocumentProcessingUnit eProcessingUnit);

#ifdef __cplusplus
}
#endif

#endif
